# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the predefined skills supplied externally

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE TO FIND MISSING PREDICATE ===
        # Feedback indicates there may be a missing predicate that blocks planning.
        # We systematically check predicates like `identified`, `lock-known`, etc., using exploration skills.

        print("===== [Exploration Phase] Checking for missing predicates via exploration skills =====")
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Collect candidate objects, drawers, locations from the observation / environment
        object_names = [name for name in positions.keys() if 'object' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        handle_names = [name for name in positions.keys() if 'handle' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'table' in name or 'room' in name]

        # Fallback for locations if none detected
        if len(location_names) == 0:
            location_names = ['locA', 'locB']  # Placeholders, to be filled as needed

        # Try to move to different locations and pick/check objects and drawers, invoking skill_code skills
        # We simply iterate some basic exploration actions; errors are caught and shown for debugging

        for loc in location_names:
            try:
                # Try moving to each location
                print(f"[Exploration] Trying to move robot to: {loc}")
                # From current position to loc; we must know 'from' and 'to'.
                # As we may not track the exact current position, we just try from each location to another.
                for from_loc in location_names:
                    if from_loc == loc:
                        continue
                    try:
                        obs, reward, done = execute_go(env, task, from_loc, loc)
                        print(f"Moved robot from {from_loc} to {loc}")
                    except Exception as e:
                        print(f"(Ignored) execute_go from {from_loc} to {loc} failed: {e}")
                        continue
            except Exception as e:
                print(f"(Ignored) Could not move robot to {loc}: {e}")
                continue

            # Try to pick each object at this location
            for obj in object_names:
                try:
                    print(f"[Exploration] Attempting to pick object: {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    print(f"Executed pick on {obj} at {loc}")
                    # If pick is successful, try to place in any drawer
                    for drawer in drawer_names:
                        try:
                            print(f"[Exploration] Attempting to place object: {obj} in drawer: {drawer} at {loc}")
                            obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            print(f"Placed {obj} into {drawer} at {loc}")
                        except Exception as e:
                            print(f"(Ignored) execute_place on {obj} in {drawer} at {loc}: {e}")
                except Exception as e:
                    print(f"(Ignored) execute_pick on {obj} at {loc}: {e}")

            # Try drawer actions: pull and push (open/close)
            for drawer in drawer_names:
                # For pulling, need a handle. We attempt to find one associated with this drawer.
                candidate_handles = [h for h in handle_names if drawer in h]
                if len(candidate_handles) == 0 and len(handle_names) > 0:
                    candidate_handles = [handle_names[0]]  # Fallback: use the first handle

                for handle in candidate_handles:
                    try:
                        print(f"[Exploration] Attempting to pull drawer: {drawer} with handle: {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"Pulled (opened) {drawer} with {handle} at {loc}")

                        # Now, immediately attempt to push (close) the drawer again
                        try:
                            print(f"[Exploration] Attempting to push (close) drawer: {drawer} at {loc}")
                            obs, reward, done = execute_push(env, task, drawer, loc)
                            print(f"Pushed (closed) {drawer} at {loc}")
                        except Exception as e:
                            print(f"(Ignored) execute_push on {drawer} at {loc}: {e}")
                    except Exception as e:
                        print(f"(Ignored) execute_pull on {drawer} with {handle} at {loc}: {e}")

            # Try sweep action for each object (if makes sense)
            for obj in object_names:
                try:
                    print(f"[Exploration] Attempting to sweep object: {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"(Ignored) execute_sweep on {obj} at {loc}: {e}")

        # Try gripper action, just in case any initial gripper state predicate missing
        try:
            print(f"[Exploration] Attempting gripper action.")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"(Ignored) execute_gripper failed: {e}")

        print("===== [Exploration Completed] Now ready for planning with new predicates discovered via actions =====")

        # === MAIN TASK PLAN (ORACLE PLAN) WOULD BE EXECUTED HERE ===
        # You should insert execution logic for the actual plan steps here, relying only on the available skills.
        # For example:
        # obs, reward, done = execute_go(env, task, <from>, <to>)
        # obs, reward, done = execute_pick(env, task, <obj>, <loc>)
        # obs, reward, done = execute_place(env, task, <obj>, <drawer>, <loc>)
        #
        # Provide your actual plan steps here based on your specific goal/problem, making
        # sure to stick to the available skills.

        print("===== [Plan Execution Placeholder] Insert oracle plan execution here using only skill_code skills =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
