# run_skeleton_task.py (Completed Executable Code)
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ---- Exploration Phase: Identify Missing Predicate ----
        # Feedback says: (drawer-closed drawer1) is the missing predicate,
        # so we need to check if the drawer is closed in the environment.
        # We'll use the available skills to check the state and try to discover it.

        print("[Exploration] Checking for drawer closed state using available skills.")

        # Suppose positions contains something like {'drawer1': {...}, ...}
        drawer_name = 'drawer1'
        drawer = positions.get('drawer1', None)
        if drawer is None:
            print("[Error] drawer1 not found in positions!")
        else:
            # We'll try to "explore" the state using available skills.
            try:
                # Before manipulating, ensure we are at the correct location.
                robot_at_pos = None
                for k, v in positions.items():
                    if k.startswith('robot'):
                        robot_at_pos = v.get('location', None)
                        break
                # Fallback: assume some location key is available
                if robot_at_pos is None and 'robot_at' in positions:
                    robot_at_pos = positions['robot_at']

                # For robustness, check where the drawer is located
                drawer_loc = drawer.get('location', None)
                if drawer_loc is None and 'drawer1_loc' in positions:
                    drawer_loc = positions['drawer1_loc']

                if robot_at_pos is not None and drawer_loc is not None and robot_at_pos != drawer_loc:
                    print(f"[Exploration] Moving robot from {robot_at_pos} to {drawer_loc}")
                    obs, reward, done = execute_go(env, task, robot_at_pos, drawer_loc)
                    robot_at_pos = drawer_loc
                else:
                    print(f"[Exploration] Robot already at drawer location: {drawer_loc}")

                # At this point, robot is at drawer location.
                # Try to push (close) the drawer. If the predicate is missing,
                # the skill will likely fail, revealing the missing (drawer-closed ...) predicate.

                # Check if drawer is open first (simulate the logic)
                is_open = drawer.get('open', None)
                is_closed = drawer.get('closed', None)
                if is_open:
                    # Try to close it
                    print("[Exploration] Drawer is open; Try to close it.")
                    try:
                        obs, reward, done = execute_push(env, task, drawer_name, drawer_loc)
                        print("[Exploration] execute_push called on drawer1.")
                    except Exception as e:
                        print(f"[Exploration] Exception while trying to push drawer: {e}")
                elif is_closed:
                    print("[Exploration] Drawer is already closed, as expected.")
                else:
                    # State unknown, try to close anyway to trigger error if missing
                    try:
                        print("[Exploration] Drawer state uncertain; trying to close to discover predicates.")
                        obs, reward, done = execute_push(env, task, drawer_name, drawer_loc)
                        print("[Exploration] execute_push called on drawer1 (unknown state).")
                    except Exception as e:
                        print(f"[Exploration] Exception (missing predicate?): {e}")
                        print("[Exploration] Likely missing predicate: (drawer-closed drawer1)")

            except Exception as e:
                print(f"[Exploration] Unexpected exception: {e}")

        # -- Main Task Plan (Example) --
        # At this point, we've attempted to identify the missing (drawer-closed drawer1) predicate via exploration.
        # Below would follow the plan actions to solve the actual user task, using only available skills.
        # Since the oracle plan is unspecified, here we end the exploration demo.

        print("===== Exploration Finished. If there was an exception or precondition failure above, the missing predicate is (drawer-closed drawer1) as per feedback. =====")
    
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
