# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Begin Exploration Phase (to resolve missing predicate)
        # The feedback shows use of (robot-at floor), but our main domain only contains
        # (robot-at ?p - location), not (robot-at ?r ?loc). Also, the exploration domain contains:
        # (robot-at ?r ?loc), as well as predicates like (at ?obj ?loc).
        # We need to perform an exploration step to see if the simulation/environment is missing
        # a predicate, e.g., (robot-at floor) is not found in the main domain's initial state.

        # We try to perform navigation across all reachable locations in the scene
        # to "explore" predicates of type (robot-at ...), check status, and identify any predicate not matching the PDDL.

        # For demonstration, we'll go through all known locations and move the robot there,
        # using execute_go (if available), and print the result to help identify any missing predicates.

        # We'll assume the object_positions dict contains location keys (common in many RLBench environments).
        # Example: {'floor': ..., 'drawer': ...}
        # List all unique locations in the environment
        location_names = []
        for name in positions:
            if 'location' in name or name == 'floor' or 'drawer' in name:
                location_names.append(name)
        # If none found, just collect all keys
        if not location_names:
            location_names = list(positions.keys())

        # If the robot's current location is known, use it; otherwise, start with the first.
        current_location = None
        for loc in location_names:
            # Sample check: does the observation or environment tell us the robot starts here?
            # As we have no direct initial predicates, we can't infer with certainty.
            # We'll default to the first location.
            current_location = loc
            break

        # Exploration: Try moving robot to each location, using execute_go.
        # This will help us deduce if any "robot-at" predicate is missing from the state.
        for dest_location in location_names:
            if dest_location == current_location:
                continue
            try:
                print(f"[Exploration] Attempting to move robot from {current_location} to {dest_location} ...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_location,
                    to_location=dest_location,
                    # Pass any additional parameters if required by execute_go
                )
                print(f"[Exploration] Success. Robot should now be at {dest_location}.")
                # After moving, set current_location to dest_location
                current_location = dest_location
                # Here, you might check if the robot's state reflects the new location;
                # if not, it's likely the predicate for (robot-at ...) is missing or mismatched
            except Exception as e:
                print(f"[Exploration] Failed to move robot from {current_location} to {dest_location}: {e}")

        # For further exploration: Try all pick/place/push/pull to see if additional precondition/effect mismatches happen.
        # This helps to identify not just the robot-at predicates but others (e.g. holding, on-floor, in-drawer, etc)

        # For each object, try to pick it up (if on the floor and hand is empty and at location)
        for obj_name in positions:
            if 'object' in obj_name or 'obj' in obj_name or 'cube' in obj_name or 'ball' in obj_name:
                try:
                    print(f"[Exploration] Attempting to pick up {obj_name} at {current_location} ...")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj_name,
                        location=current_location,
                        # Any additional parameters as required
                    )
                    print(f"[Exploration] Success picking up {obj_name}.")
                    # You may wish to attempt a place or put-down action here as well
                except Exception as e:
                    print(f"[Exploration] Failed to pick up {obj_name} at {current_location}: {e}")

        # === End of Exploration Phase ===
        # Now print a summary or instructions to check initial state and predicate coverage

        print("[Exploration] Exploration phase complete. Please review above logs to identify any missing predicate, especially 'robot-at floor', in your problem or domain PDDL.")

        # (At this point, the oracle task plan would execute the rest of its goal-oriented steps.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()