# run_skeleton_task.py (EXPLORATION + PLAN EXECUTION)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills (do not redefine!)

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset environment / task state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve positions/info
        positions = get_object_positions()

        # Predefined object_list (from feedback/example)
        object_list = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "trash",
            "trash_bin"
        ]

        # === Exploration Phase for Missing Predicate ===
        # The task may fail because a required predicate is unobservable or missing:
        # Possible predicates to determine: identified(?obj), weight-known(?obj), durability-known(?obj), lock-known(?obj), temperature-known(?obj).
        # Use only available skills, and try to "explore" with execute_pick, execute_pull, etc.

        missing_predicate_guessed = False
        for obj_name in object_list:
            # Check if object is in positions (some objects may not be spawned in scenario)
            if obj_name not in positions:
                print(f"[Exploration] Object {obj_name} not present in positions, skipping.")
                continue

            obj_pos = positions[obj_name]
            print(f"[Exploration] Exploring object: {obj_name} at position {obj_pos}")

            # Try: Use execute_pick to check weight/durability
            try:
                obs, reward, done = execute_pick(
                    env, task,
                    target_name=obj_name,
                    target_pos=obj_pos,
                    approach_distance=0.12,
                    max_steps=70,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=12.0
                )
                print(f"[Exploration:execute_pick] Picked {obj_name} -- check weight/durability inferred.")
                missing_predicate_guessed = True
                # If successful, place back/finalize via execute_place if appropriate
                if "drawer" in obj_name:
                    place_target = "drawer_top_place_left"
                    if place_target in positions:
                        obs, reward, done = execute_place(
                            env, task,
                            target_name=obj_name,
                            drawer_name="drawer_top",
                            target_pos=positions[place_target],
                            approach_distance=0.12,
                            max_steps=70,
                            threshold=0.01,
                            timeout=12.0
                        )
                elif "trash" in obj_name:
                    # Place trash into trash_bin
                    if "trash_bin" in positions:
                        obs, reward, done = execute_place(
                            env, task,
                            target_name=obj_name,
                            drawer_name="trash_bin",
                            target_pos=positions["trash_bin"],
                            approach_distance=0.12,
                            max_steps=70,
                            threshold=0.01,
                            timeout=12.0
                        )
            except Exception as e:
                print(f"[Exploration:execute_pick] Could not pick {obj_name}: {str(e)}")

            # Try: Use execute_pull for handles to check lock-known predicate
            if "handle" in obj_name:
                try:
                    obs, reward, done = execute_pull(
                        env, task,
                        drawer_name=obj_name,
                        handle_name=obj_name,
                        target_pos=obj_pos,
                        approach_distance=0.12,
                        max_steps=70,
                        threshold=0.01,
                        timeout=12.0
                    )
                    print(f"[Exploration:execute_pull] Pulled {obj_name} -- lock-known may be inferred.")
                    missing_predicate_guessed = True
                except Exception as e:
                    print(f"[Exploration:execute_pull] Could not pull {obj_name}: {str(e)}")
            # Try: Use execute_push for drawers to test open/close
            if "drawer" in obj_name:
                try:
                    obs, reward, done = execute_push(
                        env, task,
                        drawer_name=obj_name,
                        target_pos=obj_pos,
                        approach_distance=0.12,
                        max_steps=70,
                        threshold=0.01,
                        timeout=12.0
                    )
                    print(f"[Exploration:execute_push] Pushed {obj_name} -- drawer state may be observable.")
                    missing_predicate_guessed = True
                except Exception as e:
                    print(f"[Exploration:execute_push] Could not push {obj_name}: {str(e)}")
        if not missing_predicate_guessed:
            print("[Exploration] Could not infer any new predicate, you may need environment/model update.")

        # === MAIN TASK PLAN: Robust Object Validation & Manipulation ===

        # EXAMPLE OBJECT TO INTERACT WITH (to be put in a drawer)
        object_to_put_in_drawer = "trash"  # As in feedback; change here for generality

        # Validation: check if object is in defined object_list and in scene
        if object_to_put_in_drawer not in object_list:
            print(f"Error: {object_to_put_in_drawer} is not in the object_list.")
        elif object_to_put_in_drawer not in positions:
            print(f"Error: {object_to_put_in_drawer} is not in the environment positions (not spawned?)")
        else:
            obj_pos = positions[object_to_put_in_drawer]
            try:
                print(f"[Task] Picking up {object_to_put_in_drawer} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env, task,
                    target_name=object_to_put_in_drawer,
                    target_pos=obj_pos,
                    approach_distance=0.12,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Task] Successfully picked {object_to_put_in_drawer}")
            except Exception as e:
                print(f"[Task] Failed to pick {object_to_put_in_drawer}: {str(e)}")
                return

            # Assume placing in drawer_top, validate drawer exists
            drawer_name = "drawer_top"
            drawer_pos = positions.get("drawer_top_place_left", None)
            if drawer_pos is None:
                print("[Task] drawer_top_place_left is not present in positions.")
                return

            try:
                print(f"[Task] Attempting to place {object_to_put_in_drawer} in {drawer_name} at {drawer_pos}")
                obs, reward, done = execute_place(
                    env, task,
                    target_name=object_to_put_in_drawer,
                    drawer_name=drawer_name,
                    target_pos=drawer_pos,
                    approach_distance=0.12,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Successfully placed {object_to_put_in_drawer} in {drawer_name}")
            except Exception as e:
                print(f"[Task] Failed to place {object_to_put_in_drawer} in {drawer_name}: {str(e)}")
                return

        # If more actions required (multi-step oracle plan): extend stepwise using available skills as above,
        # always validating objects are in object_list and positions, and handle errors gracefully.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
