# run_skeleton_task.py (Fully Generic Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with state exploration'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE: Discover Missing Predicate via Exploration ===
        # From feedback: (robot-at dice_pose)
        # The task likely requires finding a predicate mapping or possible missing localization.

        exploration_success = False

        try:
            # Try to move the robot to 'dice_pose' location if available
            # We assume positions contains key 'dice_pose' or equivalent mapping
            if 'dice_pose' in positions:
                dice_pose = positions['dice_pose']
            else:
                # Try to infer a location similar to dice_pose
                possible_locations = [k for k in positions.keys() if 'dice' in k and 'pose' in k]
                if possible_locations:
                    dice_pose = positions[possible_locations[0]]
                else:
                    raise KeyError('dice_pose not found in object positions.')

            # Now, attempt to execute the "go" skill/action to move to dice_pose
            current_robot_location = None
            for k, v in positions.items():
                if 'robot' in k and 'pose' in k:
                    current_robot_location = v
                    break
            if not current_robot_location:
                # Fallback: try to use any 'robot' key position
                robot_keys = [k for k in positions if 'robot' in k]
                if robot_keys:
                    current_robot_location = positions[robot_keys[0]]
                else:
                    # If not found, use the origin or fallback as starting
                    current_robot_location = [0,0,0]

            print("[Exploration] Attempting to execute_go from", current_robot_location, "to", dice_pose)
            # execute_go takes two locations (from, to)
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_robot_location,
                to_location=dice_pose,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] execute_go done, robot at dice_pose.")
            else:
                print("[Exploration] execute_go finished, check state.")

            # After this, we can check whether the new predicate holds (e.g., robot-at dice_pose)
            # Exploration success (simulate effect of adding (robot-at dice_pose))
            exploration_success = True

        except Exception as e:
            print("[Exploration] Exploration phase failed:", str(e))

        # === POST-EXPLORATION: Rest of the task logic ===
        if exploration_success:
            print("[Exploration] Exploration phase successful: (robot-at dice_pose) predicate is now satisfied.")
        else:
            print("[Exploration] Exploration phase NOT successful. Predicate may be missing.")

        # === Example Plan Steps (replace with oracle plan as provided) ===
        # This is a placeholder. You would in practice loop over the oracle plan actions.
        # For demonstration, here is a hypothetical sequence using available skills:

        # Pick an object on the floor (if present in positions)
        floor_objects = [k for k in positions.keys() if 'obj' in k or 'cube' in k or 'dice' in k]
        for obj_name in floor_objects:
            try:
                obj_pos = positions[obj_name]
                print(f"[Plan] Attempting to pick object: {obj_name} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj_name,
                    object_pos=obj_pos,
                    approach_distance=0.12,
                    max_steps=80,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Plan] Successfully picked {obj_name}")
                else:
                    print(f"[Plan] Pick action finished for {obj_name}")

                # Stopping after the first successful pick for demonstration
                break
            except Exception as e:
                print(f"[Plan] Error picking {obj_name}:", str(e))

        # More plan steps would go here, depending on task definition and oracle plan

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()