# run_exploration_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now including predicate/missing knowledge exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()  # Ex: {'drawer1': (x,y,z), ...}

        # ------ Exploration Phase ------
        print("[Exploration] Starting to check for missing predicates...")

        # Based on feedback: (drawer-unlocked drawer1)
        # Assume we need to check if drawer1 is unlocked or locked.
        # In this exploration, we simulate using the available skills to
        # check (by effect or try) the lock status via execute_pull on the drawer's handle.

        # Try to identify the drawer and its handle position
        drawer_name = 'drawer1'
        handle_name = None
        for obj_name in positions.keys():
            if 'handle' in obj_name and drawer_name in obj_name:
                handle_name = obj_name
                break
        if not handle_name:
            # Use heuristic or fallback for handle detection
            handle_name = 'drawer1_handle' if 'drawer1_handle' in positions else None

        # Location handling: let’s pick an arbitrary location or from obs if available
        # Fallback: if no location, use a default or the robot's current location
        robot_location = None
        for key in positions:
            if 'robot' in key:
                robot_location = positions[key]
                break
        else:
            # fallback, if not found
            robot_location = (0, 0, 0)

        # Try exploration: attempt to "pull" (open) the drawer
        # This checks if it's unlocked (i.e., predicate is true)
        try:
            if handle_name and drawer_name in positions:
                print(f"[Exploration] Attempting to pull the drawer '{drawer_name}' using handle '{handle_name}'...")
                
                # Step 1: Move robot to drawer location if needed
                # Here we assume locations are position tuples, or string names depending on interface
                # If available, use string names
                try:
                    obs, reward, done = execute_go(
                        env, 
                        task, 
                        from_location=robot_location,
                        to_location=drawer_name,
                        max_steps=100
                    )
                    robot_location = drawer_name  # Update robot's loc to destination
                except Exception as e:
                    print(f"[Exploration] Could not execute_go to {drawer_name}: {e}")

                # Step 2: Pick the handle
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=handle_name,
                        location=robot_location,
                        approach_distance=0.10,
                        max_steps=50,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=5.0
                    )
                    print(f"[Exploration] Successfully picked handle '{handle_name}'.")
                except Exception as e:
                    print(f"[Exploration] Could not pick handle '{handle_name}': {e}")

                # Step 3: Try to pull the drawer open (this checks the predicate)
                try:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=drawer_name,
                        handle_name=handle_name,
                        location=robot_location,
                        max_steps=50,
                        threshold=0.01,
                        timeout=5.0
                    )
                    print(f"[Exploration] Pull action succeeded: '{drawer_name}' was unlocked (predicate found).")
                    # At this point, we've confirmed the drawer was unlocked: (drawer-unlocked drawer1)
                except Exception as e:
                    print(f"[Exploration] Pull action failed: '{drawer_name}' may be locked or missing predicate: {e}")
                    print("[Exploration] Missing predicate for drawer lock/unlock detected or other issue present.")

            else:
                print(f"[Exploration] Could not determine handle or drawer in positions dict. Exploration incomplete.")

        except Exception as exploration_exception:
            print(f"[Exploration] Unexpected error during exploration: {exploration_exception}")

        # ---------------- Plan Execution (after exploration) ----------------

        # TODO: Here, proceed with the oracle plan, now that you've tested/established (drawer-unlocked drawer1).
        
        # Example:
        # (Assume plan is: move to drawer, pull open, pick object, place, etc.)
        # For now, only outline as comments:
        #
        # # Move to drawer
        # obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=drawer_name, max_steps=100)
        # # Pull open (already tried above)
        # # Pick up object from floor
        # obs, reward, done = execute_pick(env, task, object_name='object1', location=drawer_name, ...)
        # # Place in drawer
        # obs, reward, done = execute_place(env, task, object_name='object1', drawer_name=drawer_name, location=drawer_name, ...)

        print("[Task] Exploration and main plan sequence complete (see above for execution details).")

    finally:
        # Ensure environment is always shut down
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
