# run_skeleton_task.py (Completed version with exploration for missing predicate discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use the predefined primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE ===
        # Feedback: (robot-at dice1_pose) is appearing, but is a missing predicate.
        # The purpose is to perform exploration to discover/verify which predicates are missing,
        # e.g. explore all positions that could correspond to 'robot-at' and test navigation.

        # We'll attempt to navigate to all known locations to check which predicates 
        # are relevant, by using the execute_go skill and observing the effect.

        # Try all pairs of locations in object_positions and see if navigation (robot-at) can be validated.
        # This allows checking if 'robot-at' at a given location is a missing predicate in the observation.

        # Find all positions that correspond to potential locations
        location_names = []
        for key in positions.keys():
            if key.endswith("_pose") or "loc" in key or "location" in key:
                location_names.append(key)
        # Fallback: if no "_pose" locations found, consider all position keys
        if not location_names:
            location_names = list(positions.keys())

        # Get current robot location from observation; fallback: first location
        current_location = None
        for key in location_names:
            if "robot" in key:
                current_location = key
                break
        if not current_location and location_names:
            current_location = location_names[0]

        # If current location not directly found, we assume first is start
        if not current_location:
            # fallback: try guess from description or initial state, but here we just pick first
            current_location = location_names[0] if location_names else None

        explored_locations = set()
        print("[Exploration] Starting exploration of all reachable locations...")

        # Try execute_go between all different locations using the available skill
        for loc in location_names:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Moving robot from {current_location} to {loc} to test 'robot-at {loc}' predicate.")
                # Call the move-related skill; as per skills list, use execute_go
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_location,
                    to_location=loc,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                explored_locations.add(loc)
                current_location = loc  # Update current location after move
                if done:
                    print("[Exploration] Task ended during exploration!")
                    break
            except Exception as e:
                print(f"[Error] Failed to move from {current_location} to {loc}: {e}")

        print(f"[Exploration] Explored locations: {explored_locations}")

        # After navigation, the environment should now have visited locations, and any missing
        # 'robot-at' predicates will be surfaced through feedback or detected.

        # === Task Plan Execution Phase (Oracle Plan) ===
        # At this point, execute_go ensures the robot can visit locations such as dice1_pose,
        # and the missing predicate is detectable.
        #
        # Here would follow the actual plan to reach the goal using the available skills.
        # Since the skeleton focuses on exploration prompted by feedback, we stop here.
        #
        # Example for further oracle plan (if you had full plan steps):
        #
        #   obs, reward, done = execute_pick(env, task, ...)  
        #   obs, reward, done = execute_place(env, task, ...) 
        #
        # ...etc...
        #
        # but as per instruction, only exploration is required here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
