# run_skeleton_task.py (Completed with Exploration Phase Logic)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all predefined skills as required.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation—with exploration phase for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Retrieve a dictionary with current object positions and state info.
        positions = get_object_positions()

        # ---------------------- EXPLORATORY PHASE ------------------------
        # Problem: Feedback indicated a missing predicate in planning/preconditions:
        #           (drawer-full drawer1)
        # Goal: Explore/interact so the agent can correctly identify 'drawer-full'
        #
        # We need to use the available skills (including exploration-type actions)
        # and perform steps to reveal that a certain drawer (e.g., drawer1)
        # is 'drawer-full'. We use only predefined skills.
        #
        # For illustration, assume object and location names commonly used:
        # - drawer1 (drawer)
        # - handle1 (handle object for drawer1)
        # - Some objects (obj1, obj2, ...) that can be loaded into the drawer.
        # - Locations: location_a, location_b, location_drawer1, etc.
        #
        # The exploration plan:
        #   1. Ensure robot is at the location of the drawer/handle.
        #   2. Ensure robot is not holding anything (hand empty).
        #   3. Pick up the handle and try to open (pull) the drawer with execute_pull.
        #   4. If drawer is not open and cannot be opened, it may be locked/full.
        #   5. Try to place an object into the drawer with execute_place; if the action fails due to 'drawer-full', we have confirmed the predicate.
        #
        # Note: All actions must be invoked via the provided skill functions.

        # ---- Setup Names ----
        # Keys come from positions dictionary.
        drawer_name = None
        handle_name = None
        drawer_loc = None

        # Discover drawer and handle names from environment info
        for k in positions.keys():
            if 'drawer' in k.lower() and drawer_name is None:
                drawer_name = k
            if 'handle' in k.lower() and handle_name is None:
                handle_name = k
        # Fallback names if not found
        if drawer_name is None:
            drawer_name = 'drawer1'
        if handle_name is None:
            handle_name = 'handle1'
        # Get location of drawer if available
        try:
            drawer_loc = positions[drawer_name]['location']
        except Exception:
            drawer_loc = 'drawer_area'  # fallback

        print(f"[Explore] Using drawer: {drawer_name}, handle: {handle_name}, at location: {drawer_loc}")

        # Find any on-floor object to try to place
        object_to_place = None
        for k, v in positions.items():
            if v.get('on_floor', False):
                object_to_place = k
                object_loc = v.get('location', 'floor_area')
                break
        if object_to_place is None:
            # fallback: just pick a random object (shouldn't happen unless empty scene)
            for k, v in positions.items():
                if v.get('type', '') == 'object':
                    object_to_place = k
                    object_loc = v.get('location', 'floor_area')
                    break

        if object_to_place is None:
            print("[Error] No suitable object to place found in environment!")
            return

        # ==============================================
        # 1. Go to drawer location (or wherever handle is)
        try:
            # Find robot's current location if available
            robot_loc = None
            for k, v in positions.items():
                if v.get('type') == 'robot':
                    robot_loc = v.get('location')
                    break
            if robot_loc is None:
                robot_loc = 'start'
            if robot_loc != drawer_loc:
                print(f"[Explore] Moving robot from {robot_loc} to {drawer_loc}")
                obs, reward, done = execute_go(env, task, robot_loc, drawer_loc)
        except Exception as e:
            print(f"[Warning] Failed to move robot: {e}")

        # 2. Ensure hand is empty (try to drop/release if not)
        # We'll run 'execute_gripper' as a "reset/free hand" skill.
        try:
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Warning] Failed to call execute_gripper: {e}")

        # 3. Try to pick up the handle to open the drawer (drawer must not be locked)
        try:
            print(f"[Explore] Attempting to pick up handle: {handle_name}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_loc)
        except Exception as e:
            print(f"[Explore] (Expected) Could not pick up handle: {e}")

        # 4. Try to pull/open the drawer with the handle
        try:
            print(f"[Explore] Trying to execute_pull for {drawer_name} using {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_loc)
        except Exception as e:
            print(f"[Explore] (Expected) Could not pull/open drawer: {e}")

        # 5. Try to place an object in the drawer (should fail if drawer is full)
        # Must pick up the object first, then place while at drawer location
        try:
            # Try to pick up an object to place
            print(f"[Explore] Picking up object to test drawer fullness: {object_to_place}")
            obs, reward, done = execute_pick(env, task, object_to_place, object_loc)
        except Exception as e:
            print(f"[Explore] (Expected) Could not pick object: {e}")

        try:
            # Try to open drawer forcibly if needed
            # (We already tried above)
            # Now, attempt place.
            print(f"[Explore] Trying to execute_place into {drawer_name}")
            obs, reward, done = execute_place(env, task, object_to_place, drawer_name, drawer_loc)
            print("[Explore] Placement succeeded: Drawer was NOT full.")
        except Exception as e:
            print(f"[Explore] Could not place object into drawer: {e}")
            print("[Observation] Based on the failure, the predicate (drawer-full {drawer_name}) is likely True.")

        # The above phase is for *exploration* only; after this, the actual plan logic may proceed.
        # In an actual workflow, we would now use this new knowledge to formulate the correct plan.

        # ==============================================
        # At this point, task-specific logic should follow, using only allowed skills.
        # For demonstration, we finish the exploration when 'drawer-full' predicate is established.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()