# run_skeleton_task.py (Executable with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine primitives like move, pick, place, etc.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ... }
        positions = get_object_positions()
        
        # === Exploration Phase: Determine Missing Predicate (as per feedback) ===
        # Since planning times out, try to discover which state information,
        # like lock-known, temperature-known, durability-known, etc., is needed.
        # We use the available skills for exploration. Try each and see if the system
        # can progress. If a skill fails, try the next exploratory skill.

        # We assume for demonstration purposes object/location/handle names.
        # In practice, retrieve them from `positions`/description or define appropriately.
        # You may need to adapt these names according to your environment object naming.
        objects = [obj for obj in positions.keys() if 'drawer' not in obj and 'handle' not in obj]
        drawers = [obj for obj in positions.keys() if 'drawer' in obj]
        handles = [obj for obj in positions.keys() if 'handle' in obj]
        locations = [loc for loc in positions.keys() if 'location' in loc or 'room' in loc or 'drawer' in loc]  # broadened to cover likely possible names

        current_location = None
        robot_free = True
        hand_empty = True
        holding_obj = None

        # Try to infer robot's initial location from descriptions or obs if provided
        for k in positions:
            if "robot" in k or "Robot" in k:
                current_location = positions[k]  # Might use key or value depending on format

        # Fallback: pick a location
        if not current_location and locations:
            current_location = positions[locations[0]] if locations[0] in positions else None

        # Record exploration results
        exploration_log = {}

        # Try all available exploration-related skills (from oracle plan/feedback):
        skill_sequence = [
            "execute_go_identify",
            "execute_go_temperature",
            "execute_pick_weight",
            "execute_pick_durability",
            "execute_pull"
        ]

        # Available skill names in this codebase
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep',
            'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For feedback, try to find the missing predicate with only exploration knowledge
        # and available skills. If exploration fails or times out, continue to the regular plan.

        # Set up example parameter sets for each action using detected objects/handles/drawers/locations.
        try:
            # --- 1. Try go/identify at each drawer and object location ---
            for to_location in locations:
                if "drawer" in to_location or any(drawer in to_location for drawer in drawers):
                    # Try execute_go (since execute_go_identify is not a provided skill)
                    if 'execute_go' in available_skills:
                        try:
                            print(f"[Exploration] Trying execute_go to '{to_location}'...")
                            obs, reward, done = execute_go(env, current_location, to_location)
                            print(f"[Exploration] Robot moved from {current_location} to {to_location}.")
                            current_location = to_location
                            exploration_log["execute_go"] = True
                        except Exception as e:
                            print(f"[Exploration] execute_go to {to_location} failed: {e}")
                            exploration_log["execute_go"] = False
                
                # Try execute_sweep to see if floor objects can be revealed or manipulated
                for obj in objects:
                    if 'execute_sweep' in available_skills:
                        try:
                            print(f"[Exploration] Trying execute_sweep on '{obj}' at {to_location} ...")
                            obs, reward, done = execute_sweep(env, obj, to_location)
                            print(f"[Exploration] Swept {obj} at {to_location}")
                            exploration_log[f"execute_sweep_{obj}_{to_location}"] = True
                        except Exception as e:
                            print(f"[Exploration] execute_sweep for {obj} at {to_location} failed: {e}")
                            exploration_log[f"execute_sweep_{obj}_{to_location}"] = False
            
            # --- 2. Try picking up every object ---
            for obj in objects:
                if 'execute_pick' in available_skills and hand_empty and robot_free:
                    try:
                        print(f"[Exploration] Trying execute_pick on '{obj}' at {current_location} ...")
                        obs, reward, done = execute_pick(env, obj, current_location)
                        print(f"[Exploration] Picked up {obj} at {current_location}")
                        hand_empty = False
                        robot_free = False
                        holding_obj = obj
                        exploration_log[f"execute_pick_{obj}"] = True
                    except Exception as e:
                        print(f"[Exploration] execute_pick for {obj} at {current_location} failed: {e}")
                        exploration_log[f"execute_pick_{obj}"] = False

            # --- 3. Try pulling drawer handles with execute_pull (lock-known exploration) ---
            for drawer, handle in zip(drawers, handles):
                if 'execute_pull' in available_skills and not hand_empty and holding_obj == handle:
                    try:
                        print(f"[Exploration] Trying execute_pull on '{drawer}' using handle '{handle}' at {current_location} ...")
                        obs, reward, done = execute_pull(env, drawer, handle, current_location)
                        print(f"[Exploration] Pulled {drawer} using {handle}")
                        robot_free = True
                        hand_empty = True
                        holding_obj = None
                        exploration_log[f"execute_pull_{drawer}_{handle}"] = True
                    except Exception as e:
                        print(f"[Exploration] execute_pull for {drawer} with {handle} at {current_location} failed: {e}")
                        exploration_log[f"execute_pull_{drawer}_{handle}"] = False
            
            # --- 4. Try pushing drawers closed ---
            for drawer in drawers:
                if 'execute_push' in available_skills and robot_free:
                    try:
                        print(f"[Exploration] Trying execute_push '{drawer}' at {current_location} ...")
                        obs, reward, done = execute_push(env, drawer, current_location)
                        print(f"[Exploration] Pushed {drawer} closed at {current_location}")
                        exploration_log[f"execute_push_{drawer}"] = True
                    except Exception as e:
                        print(f"[Exploration] execute_push for {drawer} at {current_location} failed: {e}")
                        exploration_log[f"execute_push_{drawer}"] = False

            # --- 5. Try placing objects into drawers ---
            for obj in objects:
                for drawer in drawers:
                    if 'execute_place' in available_skills and not hand_empty:
                        try:
                            print(f"[Exploration] Trying execute_place: placing '{obj}' into '{drawer}' at {current_location} ...")
                            obs, reward, done = execute_place(env, obj, drawer, current_location)
                            print(f"[Exploration] Placed {obj} into {drawer}")
                            hand_empty = True
                            robot_free = True
                            holding_obj = None
                            exploration_log[f"execute_place_{obj}_{drawer}"] = True
                        except Exception as e:
                            print(f"[Exploration] execute_place for {obj} into {drawer} at {current_location} failed: {e}")
                            exploration_log[f"execute_place_{obj}_{drawer}"] = False

            # --- 6. Try rotating (if needed by environment) ---
            if 'execute_rotate' in available_skills:
                try:
                    print(f"[Exploration] Trying execute_rotate ...")
                    obs, reward, done = execute_rotate(env)
                    print("[Exploration] Executed rotate skill")
                    exploration_log["execute_rotate"] = True
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed: {e}")
                    exploration_log["execute_rotate"] = False

            # --- 7. Try gripper open/close (if skill exists and needed) ---
            if 'execute_gripper' in available_skills:
                try:
                    print(f"[Exploration] Trying execute_gripper ...")
                    obs, reward, done = execute_gripper(env)
                    print("[Exploration] Executed gripper skill")
                    exploration_log["execute_gripper"] = True
                except Exception as e:
                    print(f"[Exploration] execute_gripper failed: {e}")
                    exploration_log["execute_gripper"] = False

        except Exception as ex:
            print(f"[ERROR] Exploration phase failed: {ex}")

        print("\n[EXPLORATION RESULTS]")
        for key, value in exploration_log.items():
            print(f"{key}: {value}")

        # === END OF EXPLORATION ===
        # At this point, based on what succeeded or failed, you can inspect the log and environment
        # to determine if any predicates (e.g., lock-known) are required before the plan can proceed.
        # Use your findings to design the rest of your plan.

        print("\n[INFO] Exploration phase complete. Proceeding to normal task execution ...\n")

        # === Example Task Plan (you'd fill in these steps for the real plan) ===
        # This is still placeholder logic -- adapt to your goal.

        # For demonstration, try an execution sequence:
        # 1. Move to a drawer or location.
        # 2. Pick up a handle if available.
        # 3. Pull the drawer using the handle.
        # 4. Place an object in the drawer.
        try:
            if locations and 'execute_go' in available_skills:
                from_location = current_location
                to_location = locations[0]
                print(f"[Task] Moving from {from_location} to {to_location}")
                obs, reward, done = execute_go(env, from_location, to_location)
                current_location = to_location

            if handles and 'execute_pick' in available_skills and hand_empty:
                handle = handles[0]
                print(f"[Task] Picking up handle {handle} at {current_location}")
                obs, reward, done = execute_pick(env, handle, current_location)
                hand_empty = False
                robot_free = False
                holding_obj = handle

            if drawers and handles and 'execute_pull' in available_skills and not hand_empty and holding_obj == handles[0]:
                drawer = drawers[0]
                handle = handles[0]
                print(f"[Task] Pulling drawer {drawer} using handle {handle} at {current_location}")
                obs, reward, done = execute_pull(env, drawer, handle, current_location)
                hand_empty = True
                robot_free = True
                holding_obj = None

            if objects and drawers and 'execute_pick' in available_skills and hand_empty:
                obj = objects[0]
                print(f"[Task] Picking up object {obj} at {current_location}")
                obs, reward, done = execute_pick(env, obj, current_location)
                hand_empty = False
                robot_free = False
                holding_obj = obj

            if objects and drawers and 'execute_place' in available_skills and not hand_empty and holding_obj == objects[0]:
                obj = objects[0]
                drawer = drawers[0]
                print(f"[Task] Placing object {obj} into drawer {drawer} at {current_location}")
                obs, reward, done = execute_place(env, obj, drawer, current_location)
                hand_empty = True
                robot_free = True
                holding_obj = None

        except Exception as task_exception:
            print(f"[ERROR] During task execution: {task_exception}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
