# run_skeleton_task.py (Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine or change skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect positions dict e.g. {'robot': (..), 'drawer1': (..), ...}

        # === Exploration Phase: Identify drawer-full predicate ===
        print("[Exploration] Starting exploration to find 'drawer-full' state.")

        # This phase attempts to interact with the drawer and observe if it can become "full".
        # The skill names provided do NOT include a predicate checker, but we can attempt to
        # repeatedly use execute_place on drawer1 until we get some feedback (and log/process any error).
        # If execute_place fails because the drawer is full, we infer (drawer-full drawer1).

        # You must use only the available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this code, we wrap all placement steps in try/except and check for failure as proof of 'drawer-full'.

        # Define key object/location variables (fill in based on your environment)
        drawer_name = 'drawer1'
        handle_name = 'drawer1_handle'
        robot_location = 'home'
        drawer_location = 'drawer1_location'
        objects_on_floor = [obj for obj in positions if obj.startswith('object') or obj.startswith('obj')]

        # Internal exploration log
        max_place_attempts = 5
        placed_objects = []
        drawer_full = False

        # 1. Move robot to the drawer location if not already there
        if positions.get('robot') != positions.get(drawer_location):
            try:
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=drawer_location)
                print(f"[Exploration] Robot moved from '{robot_location}' to '{drawer_location}'.")
                robot_location = drawer_location
            except Exception as e:
                print(f"[Exploration] Failed to move robot: {e}")
                # If failed to move, cannot proceed
                return

        # 2. Open drawer if not open
        # First pick the handle if available, then pull
        try:
            obs, reward, done, info = execute_pick(env, task, object=handle_name, location=drawer_location)
            print(f"[Exploration] Picked up drawer handle '{handle_name}'.")
            obs, reward, done, info = execute_pull(env, task, drawer=drawer_name, handle=handle_name, location=drawer_location)
            print(f"[Exploration] Pulled drawer '{drawer_name}' open.")
        except Exception as e:
            print(f"[Exploration] Could not open the drawer: {e}")
            # Already open or another issue -- continue

        # 3. Place as many objects as possible into the drawer to test for 'full'
        for obj in objects_on_floor[:max_place_attempts]:
            try:
                # Pick the object
                obs, reward, done, info = execute_pick(env, task, object=obj, location=drawer_location)
                print(f"[Exploration] Picked up object '{obj}'.")

                # Place it in the drawer
                obs, reward, done, info = execute_place(env, task, object=obj, drawer=drawer_name, location=drawer_location)
                print(f"[Exploration] Placed object '{obj}' into drawer '{drawer_name}'.")
                placed_objects.append(obj)
            except Exception as e:
                # If it fails, and the drawer already has some objects, assume it's full
                print(f"[Exploration] Could NOT place '{obj}' into '{drawer_name}', likely FULL. Exception: {e}")
                drawer_full = True
                break

        if drawer_full:
            print("[Exploration] drawer-full predicate confirmed for", drawer_name)
        else:
            print("[Exploration] drawer did NOT become full after attempts.")

        # Record the discovered predicate
        discovered_predicate = f"(drawer-full {drawer_name})" if drawer_full else None

        # 4. (Optional) Clean up: close the drawer after exploration
        try:
            obs, reward, done, info = execute_push(env, task, drawer=drawer_name, location=drawer_location)
            print(f"[Exploration] Drawer '{drawer_name}' pushed closed.")
        except Exception as e:
            print(f"[Exploration] Could not push drawer closed: {e}")

        # === End Exploration: Proceed to Normal Oracle Plan Here ===
        print("[Oracle] Ready to proceed to main plan. Insert your oracle plan execution below.")
        # EXAMPLE: if you were given a plan, this is where you'd invoke the necessary skills step by step.

        # For demonstration purposes, echo the discovered predicate
        if discovered_predicate:
            print("[Exploration] MISSING PREDICATE FOUND:", discovered_predicate)

        # Example end of plan:
        print("[Task] Exploration and predicate identification complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
