# run_skeleton_task.py (Completed, Feedback-Aware, Exploration-Integrated)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally provided skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.
    Now includes an exploration phase to detect missing predicates/knowledge.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions/States ===
        # Provided utility for object positions and possibly other info
        positions = get_object_positions()

        # === EXPLORATION PHASE: Detecting missing predicates ===
        # Observation: Action plans are blocked due to (likely) missing lock/unlock or related predicate.
        # Strategy: 
        #   - Try pulling on drawers/objects or opening/closing them;
        #   - Monitor outcome/errors;
        #   - Identify if knowledge (e.g., lock-known) is missing and required.

        # Available primitive/exploration skills (from domain/feedback):
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        #  'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # Based on exploration knowledge, 'execute_pull' is used to detect (lock-known ?obj)
        # For "drawer" objects, pulling may reveal lock state. 

        # --- Example Exploration Loop ---
        # Assumptions:
        # - positions contains all objects and types are known by key/name conventions.
        # - For drawers and their handles, assume dictionary keys allow access.
        # - Skip precise type parsing for brevity and generality.

        print("[Exploration] Attempting to detect lock/unlock status of all drawers (and detect missing predicates)...")
        for obj_name, obj_info in positions.items():
            # Heuristic: look for 'drawer' in name (expand as needed)
            if 'drawer' in obj_name.lower():
                # For each drawer, try to find its handle
                handle_name = None
                for o_n in positions:
                    if 'handle' in o_n.lower() and obj_name in o_n:
                        handle_name = o_n
                        break
                if handle_name is None:
                    print(f"[Exploration] No handle found for {obj_name}, skipping...")
                    continue

                # Get robot's current location
                # Try to go to the drawer's location if not there already
                drawer_location = obj_info.get('location', None)
                try:
                    robot_location = None
                    for l in positions:
                        if 'robot' in l.lower():
                            robot_location = positions[l].get('location', None)
                            break
                    if robot_location != drawer_location and drawer_location is not None:
                        # Move the robot to drawer's location
                        print(f"[Exploration] Moving robot to location of {obj_name} ({drawer_location})...")
                        try:
                            execute_go(env, robot_location, drawer_location)
                        except Exception as e:
                            print(f"[Exploration] Error in execute_go: {e}")
                    
                    # Try to pick the handle
                    print(f"[Exploration] Trying to pick handle {handle_name} at location {drawer_location}...")
                    try:
                        # No explicit hand param (as per skill signatures)
                        obs, reward, done = execute_pick(env, handle_name, drawer_location)
                    except Exception as e:
                        print(f"[Exploration] Error in execute_pick for {handle_name}: {e}")

                    # Pull the drawer to test lock-known predicate
                    print(f"[Exploration] Attempting execute_pull on {obj_name} (handle {handle_name})...")
                    try:
                        # Skill signature includes (drawer, handle, location)
                        obs, reward, done = execute_pull(env, obj_name, handle_name, drawer_location)
                        print(f"[Exploration] Pulled {obj_name}. If success, likely lock/unlock info revealed.")
                    except Exception as e:
                        print(f"[Exploration] Error on execute_pull: {e}")
                        # If error due to predicate, lock-known or drawer-locked may be missing!
                except Exception as e:
                    print(f"[Exploration] General exploration error for {obj_name}: {e}")

        print("[Exploration] Exploration phase complete.")

        # === TASK PHASE: Run Oracle Plan (example logic) ===
        # After exploration, you would typically run the oracle plan.
        # Here, using only the available skills, we'll try a hypothetical typical sequence
        # for interacting with a drawer (pick object, open drawer, place object).

        # Find a movable object and a drawer to interact with
        movable_obj = None
        movable_loc = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('type', '').lower() == 'object' or 'movable' in obj_name.lower():
                movable_obj = obj_name
                movable_loc = obj_info.get('location', None)
                break
        # Find a drawer and its handle/location
        drawer_name = None
        handle_name = None
        drawer_loc = None
        for obj_name, obj_info in positions.items():
            if 'drawer' in obj_name.lower():
                drawer_name = obj_name
                drawer_loc = obj_info.get('location', None)
                # Find handle
                for o_n in positions:
                    if 'handle' in o_n.lower() and drawer_name in o_n:
                        handle_name = o_n
                        break
                break

        # Now run a robust generic plan: pick object, open drawer, place object
        try:
            if movable_obj and drawer_name and handle_name:
                robot_location = None
                for l in positions:
                    if 'robot' in l.lower():
                        robot_location = positions[l].get('location', None)
                        break

                # Move to object location
                if robot_location != movable_loc and movable_loc is not None:
                    print(f"[Plan] Moving robot to object location {movable_loc}...")
                    try:
                        execute_go(env, robot_location, movable_loc)
                        robot_location = movable_loc
                    except Exception as e:
                        print(f"[Plan] execute_go failed: {e}")

                # Pick the object
                print(f"[Plan] Attempting to pick up {movable_obj} at {movable_loc}...")
                try:
                    obs, reward, done = execute_pick(env, movable_obj, movable_loc)
                except Exception as e:
                    print(f"[Plan] execute_pick failed: {e}")

                # Move to drawer location if needed
                if robot_location != drawer_loc and drawer_loc is not None:
                    print(f"[Plan] Moving robot to drawer location {drawer_loc}...")
                    try:
                        execute_go(env, robot_location, drawer_loc)
                        robot_location = drawer_loc
                    except Exception as e:
                        print(f"[Plan] execute_go to drawer failed: {e}")

                # Pull handle (open drawer)
                print(f"[Plan] Attempting to open drawer {drawer_name} via handle {handle_name}...")
                try:
                    obs, reward, done = execute_pull(env, drawer_name, handle_name, drawer_loc)
                except Exception as e:
                    print(f"[Plan] execute_pull failed: {e}")

                # Place object into drawer
                print(f"[Plan] Placing {movable_obj} into {drawer_name} at {drawer_loc}...")
                try:
                    obs, reward, done = execute_place(env, movable_obj, drawer_name, drawer_loc)
                except Exception as e:
                    print(f"[Plan] execute_place failed: {e}")
            else:
                print("[Plan] Could not identify all required objects/drawers for plan execution.")
        except Exception as e:
            print(f"[Plan] General error in plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
