# run_skeleton_task.py (Filled Out for Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Only use directly, do not redefine any skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions is expected in the format:
        # { 'drawer1': (x,y,z), 'handle1': (x,y,z), 'obj_1': (x,y,z), ...}

        # === Exploration Phase: Identify Missing Predicate via Action Execution ===
        print("[Exploration] Starting predicate exploration based on feedback.")

        # Based on feedback, 'drawer-closed drawer1' is the missing predicate;
        # we want to verify it, so we'll attempt to "explore" via an available skill.

        # Step 1: Move robot to the location near drawer1 (if not already)
        # We'll assume 'drawer1' has a known location in positions dict
        robot_loc = None
        drawer1_loc = None
        # Find robot's current location
        for k in positions:
            if "robot" in k:
                robot_loc = k
            if k == "drawer1":
                drawer1_loc = k
        if drawer1_loc is None:
            print("[Error] drawer1 not found in object positions.")
            return

        # For demonstration purposes, we assign variable names
        drawer1_position = positions['drawer1']
        # Use a location key, not the position tuple, for skills
        # (assuming keys represent location names that the low-level env understands)
        # If you know the robot's current location key, use it; else, use a default
        # Try to find a location key related to drawer1
        location_key = 'drawer1'
        
        # Try to execute execute_go to move robot to drawer's location
        try:
            # Find current location (some environments use string keys)
            available_locations = [k for k in positions.keys() if k.startswith('loc') or k.startswith('drawer')]
            if available_locations:
                current_location = available_locations[0]
                if current_location != location_key:
                    print(f"[Exploration] Moving from {current_location} to {location_key} for drawer1 predicate exploration.")
                    obs, reward, done = execute_go(env, task, current_location, location_key)
                    if done:
                        print("[Info] Task ended during movement phase.")
                        return
        except Exception as e:
            print(f"[Error] During robot movement to drawer1: {e}")
            return

        # Step 2: Attempt a skill that requires (drawer-closed drawer1) as precondition
        # According to the domain, execute_pull requires (drawer-closed drawer1)
        # We'll attempt to execute_pull handle1 to open the drawer
        
        # Find handle associated with drawer1
        handle_name = None
        for k in positions:
            if k.startswith('handle') and 'drawer1' in k:
                handle_name = k
                break
        if not handle_name:
            # Try default known handle
            handle_name = 'handle1'
            if handle_name not in positions:
                print("[Error] handle1 not found for drawer1.")
                return
        print(f"[Exploration] Attempting to pick handle {handle_name} for drawer1.")

        # Pick up the handle
        try:
            # Need to ensure hand is empty and robot free; proceed to pick handle
            obs, reward, done = execute_pick(env, task, handle_name, location_key)
            if done:
                print("[Info] Task ended during picking handle.")
                return
        except Exception as e:
            print(f"[Error] During picking handle for drawer1: {e}")
            return

        # Now, try to execute_pull (open the drawer)
        try:
            # As per PDDL, execute_pull(drawer, handle, location)
            print("[Exploration] Attempting to pull (open) the drawer using handle.")
            obs, reward, done = execute_pull(env, task, "drawer1", handle_name, location_key)
            if done:
                print("[Info] Task ended after attempting to pull drawer.")
                return
        except Exception as e:
            print(f"[Error] During execute_pull for drawer1: {e}")
            print("[Exploration] Likely missing predicate is required here, i.e., (drawer-closed drawer1).")
            return
        
        # If everything succeeded, predicate is satisfied and not actually missing

        print("[Exploration] Predicate (drawer-closed drawer1) is present and allowed skill execution.")
        print("[Exploration] If execution was blocked above, the predicate is required and missing, based on the feedback.")

        # You can insert further exploration/planning here as needed using the available skills
        # For now, Exploration phase demonstrates how to uncover missing predicates as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
