# run_skeleton_task.py (Completed for Exploration & Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *   # Use only predefined skills; do not redefine any action.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now augmented with exploration to identify missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase for Predicate Discovery ===
        # Based on feedback, we look for (drawer-unlocked drawer1) predicate.
        print("[Exploration] Start checking predicates for drawer lock state...")

        # Step 1: Identify all drawers and their associated handles (if possible)
        # Note: This part assumes you can infer drawer and handle names from descriptions or positions
        drawers = []
        handles = {}
        for obj_name in positions:
            if 'drawer' in obj_name:
                drawers.append(obj_name)
            if 'handle' in obj_name:
                # Try to associate handle to drawer based on proximity or naming convention.
                # We try to match by suffix, e.g. handle1 to drawer1, else fallback to scene info.
                for dr in drawers:
                    if dr[-1] == obj_name[-1]:
                        handles[dr] = obj_name

        # Fallback: If no specific handle, assign dummy or random one (simulation context)
        for dr in drawers:
            if dr not in handles:
                for obj_name in positions:
                    if 'handle' in obj_name:
                        handles[dr] = obj_name
                        break

        # Step 2: For each drawer, try to test its lock/unlock/open states
        for drawer in drawers:
            print(f"[Exploration] Examining: {drawer}")

            # Find the robot position and move to the drawer's location, if necessary
            robot_init_pos = None
            robot_pos_name = None
            for name, pos in positions.items():
                if 'robot' in name or 'agent' in name:
                    robot_init_pos = pos
                    robot_pos_name = name
                    break

            drawer_pos = positions[drawer] if drawer in positions else None

            # If a go/move skill is available, move the robot to the drawer location
            # Skill name in available skills is: 'execute_go'
            if drawer_pos and robot_init_pos:
                try:
                    obs, reward, done = execute_go(env, task,
                        from_pos=robot_init_pos, to_pos=drawer_pos,
                        max_steps=80, threshold=0.02
                    )
                    print(f"[Exploration] Robot moved to {drawer}.")
                except Exception as e:
                    print(f"[Exploration] Move to {drawer} failed: {e}")

            # Step 3: Try to pull (open) the drawer, using its handle; explore lock state
            handle = handles.get(drawer, None)
            if not handle:
                print(f"[Exploration] No handle found for {drawer}; skipping pull exploration.")
                continue

            # Try: Pick up the handle (must be on-floor or at a reachable location)
            handle_pos = positions[handle] if handle in positions else None
            if not handle_pos:
                print(f"[Exploration] No position found for handle {handle}.")
                continue

            # Pick handle (skill: execute_pick)
            try:
                obs, reward, done = execute_pick(env, task,
                    obj_name=handle,
                    obj_pos=handle_pos,
                    approach_distance=0.10,
                    max_steps=80,
                    threshold=0.015,
                    timeout=8.0
                )
                print(f"[Exploration] Picked up handle {handle}.")
            except Exception as e:
                print(f"[Exploration] Failed picking up handle {handle}: {e}")

            # Now, try to 'pull' the drawer with the handle (skill: execute_pull)
            # execute_pull params: (drawer, handle, location)
            try:
                obs, reward, done = execute_pull(env, task,
                    drawer_name=drawer,
                    handle_name=handle,
                    location=drawer_pos,
                    max_steps=60,
                    threshold=0.01,
                    timeout=8.0
                )
                print(f"[Exploration] Attempted to pull {drawer} (via handle {handle}).")
            except Exception as e:
                print(f"[Exploration] Failed to pull (open) {drawer}: {e}")

            # IF this fails (e.g. raising, or drawer doesn't open), infer that the missing predicate might be drawer-locked or drawer-unlocked
            print(f"[Exploration] If pulling {drawer} was unsuccessful, the predicate (drawer-unlocked {drawer}) may be missing or required for planning.")

            # Drop the handle for cleanliness using place or similar (optional, depending on domain logic).
            # Skill: execute_place (if relevant), but only if permitted in the context.

        print("[Exploration] Exploration phase complete. Please check environment logs or feedback to discover missing predicates such as (drawer-unlocked drawer1) and include/observe as needed.")

        # === End of Exploration ===

        # === Here, you would proceed with running the oracle plan, now aware of the newly discovered predicate. ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()