# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills/actions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation with exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Step 1: Exploration Phase to Detect Missing Predicate ===
        print("[Exploration] Start exploration to detect missing 'robot-at drawer-area' predicate.")

        # Oracle plan cannot proceed if robot's location is not tracked or predicate is missing.
        # Based on feedback: (robot-at drawer-area) missing → need to verify/establish robot position.

        # Try to move the robot to "drawer-area" using the available skill execute_go.
        # (Assume we know current robot location from observation or use a plausible default)
        
        # Attempt to determine robot's current and goal location
        current_robot_location = None
        goal_location = "drawer-area"
        
        # Try to find the robot's current location from the observation or positions
        try:
            if 'robot' in positions:
                current_robot_location = positions['robot'].get('location', None)
            elif 'robot-location' in positions:
                current_robot_location = positions['robot-location']
            # Fallback: check keys for any value matching a likely location
            else:
                for k in positions:
                    if k.startswith("robot-") and isinstance(positions[k], str):
                        current_robot_location = positions[k]
                        break
        except Exception as e:
            print("[Exploration] Warning: Could not determine robot's current location from positions:", e)
            
        if current_robot_location is None:
            # Best effort fallback: try default
            current_robot_location = "init-area"
            print("[Exploration] Defaulting current robot location to:", current_robot_location)

        print(f"[Exploration] Robot current location: {current_robot_location}, goal: {goal_location}")
        
        # Call the navigation/move skill to set the (robot-at drawer-area) predicate
        # Skill: execute_go
        try:
            obs, reward, done = execute_go(
                env, 
                task,
                from_location=current_robot_location, 
                to_location=goal_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Skill execute_go called to achieve (robot-at drawer-area)")
        except Exception as e:
            print(f"[Exploration] Failed to execute_go: {e}")
            # Exploration unsuccessful, cannot proceed
            print("[Exploration] Could not establish (robot-at drawer-area), stopping task.")
            return

        # Confirm predicate is now true via observation, if possible
        # (Assume task.get_observation returns dictionary with predicates)
        obs_update = None
        try:
            obs_update = task.get_observation()
            if hasattr(obs_update, "get"):
                if obs_update.get("robot-at") == goal_location or \
                   obs_update.get("robot-at-"+goal_location) is True:
                    print(f"[Exploration] Confirmed: (robot-at {goal_location}) established in observation.")
                else:
                    print(f"[Exploration] Warning: (robot-at {goal_location}) not found in observation. May still be missing.")
            else:
                print("[Exploration] Warning: Observation does not have standard predicate information.")
        except Exception as e:
            print(f"[Exploration] Could not confirm predicate in observation: {e}")

        # === Step 2: Proceed with Oracle Plan using only available skill functions ===
        # (You would replace the code below with actual plan, skills, and object names; here is a template.)
        print("[Task] Begin Oracle Plan execution (stubbed example).")
        try:
            # Example: Pick up an object from the floor at drawer-area
            target_object = None
            available_objects = [k for k,v in positions.items() if v.get('location',None) == goal_location]
            if available_objects:
                target_object = available_objects[0]
            else:
                # Fallback to any object assumed to be at 'drawer-area'
                for k,v in positions.items():
                    if 'drawer' in k and k != goal_location:
                        target_object = k
                        break

            if not target_object:
                print("[Task] No target object detected at drawer-area. Task cannot proceed.")
                return

            # Skill: execute_pick(object, location)
            obs, reward, done = execute_pick(
                env,
                task,
                o=target_object,
                p=goal_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Picked up object {target_object} at {goal_location}.")

            # Continue with other skills as needed
            # Place, push, pull, etc. (This is a stub; actual task logic depends on plan.)

            # End of stub example
            print("[Task] Oracle Plan execution step finished.")

        except Exception as e:
            print(f"[Task] Exception during plan execution: {e}")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
