# run_skeleton_task.py (Augmented for Predicate Exploration and Skill-based Execution)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine primitives. Use actions like execute_pick, execute_place, etc.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running ANY task in your simulation, with exploration for missing predicate detection.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve environment object positions
        positions = get_object_positions()

        # The exploration requirement advises us to "complete exploration code that solves the problems identified through feedback."
        # The PDDL & feedback suggests some predicate (e.g., about object, drawer, robot state) is missing or not correctly set.

        # -- EXPLORATION PHASE: Try all available skills relevant to information gathering and manipulation --
        # Use the available skills and log result, aiming to uncover what predicate or precondition is missing.
        # We avoid hardcoding, and instead, try the exploration-relevant actions (see exploration knowledge) systematically.
        # The available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # ---------------
        # Example placeholder names (replace with your actual environment's objects/locations as needed):
        #   objects: keys in 'positions' or as described by 'descriptions'
        #   locations: e.g., ['location_a', 'location_b', ...]
        #   drawers: objects with type 'drawer', found via descriptions or naming convention.
        #   handles: objects with 'handle' or in 'handle-of' relationships.

        # Assume 'descriptions' or positions dictionary gives us the full set.
        # Here we'll do a robust, data-driven enumeration:

        object_names = []
        drawer_names = []
        handle_names = []
        location_names = []

        # Scan through descriptions or positions to classify objects
        for name, info in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'location' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        if not location_names:
            # Likely locations entries are missing. As fallback, pull from position values
            for name, pose in positions.items():
                if 'loc' in name or 'table' in name or 'room' in name:
                    location_names.append(name)
            # Or treat all keys as possible locations
            if not location_names:
                location_names = list(positions.keys())

        # For the purpose of exploration (finding missing predicates), attempt:
        #   1) Moving the robot to every location (execute_go)
        #   2) Picking up each object at each location (execute_pick)
        #   3) Attempting to open/close drawers (execute_pull, execute_push)
        #   4) Placing objects into drawers (execute_place)
        #   5) Sweeping (execute_sweep), Gripper operations (execute_gripper)
        #   6) Logging what succeeded/failed for missing info

        def safe_call(action_fn, *args, **kwargs):
            try:
                obs, reward, done = action_fn(env, task, *args, **kwargs)
                print(f"[SKILL EXECUTED] {action_fn.__name__} with args={args} kwargs={kwargs} SUCCESS.")
                return (obs, reward, done)
            except Exception as e:
                print(f"[SKILL FAILED] {action_fn.__name__} args={args} kwargs={kwargs} EXCEPTION: {e}")
                return (None, None, False)

        # GUESS: The robot starts at some location
        robot_curr_loc = None
        for loc in location_names:
            # Try to record/identify the robot's current location
            if 'robot' in positions and positions['robot'] == positions[loc]:
                robot_curr_loc = loc
                break
        if not robot_curr_loc:
            # Fallback: default to first location or to 'workspace'
            robot_curr_loc = location_names[0]

        # 1) Try execute_go: move robot between locations
        for loc in location_names:
            if loc != robot_curr_loc:
                safe_call(execute_go, robot_curr_loc, loc)
                robot_curr_loc = loc

        # 2) Try picking up each object at each location
        for obj in object_names:
            # Attempt pick at every location (robot must be at that location for the skill)
            for loc in location_names:
                if loc != robot_curr_loc:
                    safe_call(execute_go, robot_curr_loc, loc)
                    robot_curr_loc = loc

                safe_call(execute_pick, obj, loc)

        # 3) Try opening/closing each drawer
        for drawer in drawer_names:
            # Try pulling (requires holding the handle and correct preconditions)
            relevant_handles = [h for h in handle_names if drawer in h or 'handle' in h]
            for handle in relevant_handles:
                # To pull, we first need to pick the handle
                for loc in location_names:
                    if loc != robot_curr_loc:
                        safe_call(execute_go, robot_curr_loc, loc)
                        robot_curr_loc = loc
                    safe_call(execute_pick, handle, loc)
                    # Try pull operation
                    safe_call(execute_pull, drawer, handle, loc)
            # Try pushing to close (robot at loc, drawer open)
            for loc in location_names:
                safe_call(execute_push, drawer, loc)

        # 4) Try placing every object into every drawer at every location
        for drawer in drawer_names:
            for obj in object_names:
                for loc in location_names:
                    safe_call(execute_place, obj, drawer, loc)

        # 5) Try sweeps and gripper
        for obj in object_names:
            for loc in location_names:
                safe_call(execute_sweep, obj, loc)
        safe_call(execute_gripper)

        print("===== Exploration phase complete. Log above shows which actions succeeded and failed, indicating possible missing predicates. =====")

        # -- At this point, use information from which skills/actions failed --
        # The exploration results and logs will suggest what predicate is missing (e.g. if all 'pull' fail due to missing 'drawer-unlocked', etc).

        # TODO: If you have an oracle plan, you can now execute it step by step with known, validated predicates/actions.
        # For now, exploration itself satisfies Task Requirement 1 for missing predicate discovery.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()