# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Predefined skill functions ONLY

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Dictionary: e.g., positions = {'drawer': (x,y,z), 'handle': (x,y,z), ...}
        positions = get_object_positions()

        # -------------------
        # Exploration Phase
        # -------------------
        print("=== [Exploration] Start ===")
        exploration_done = False

        # Based on feedback, we need to check the predicate (robot-at drawer-area)
        # Try to move the robot to 'drawer-area' using execute_go
        # This will also help us confirm the availability of 'drawer-area' and its preconditions
        
        try:
            # Check keys for locations
            location_keys = [k for k in positions.keys() if 'area' in k or 'location' in k or 'drawer' in k]
            robot_loc = None
            for k in positions.keys():
                if k.startswith("robot"):
                    robot_loc = k
                    break

            # For exploration, if both locations are present, attempt the move
            # If not, use plausible names from positions
            from_location = None
            to_location = None

            # Heuristic: Try to find a robot's current location.
            if robot_loc:
                from_location = robot_loc
            else:
                # As fallback, pick the first plausible location
                for k in positions.keys():
                    if 'area' in k or 'location' in k:
                        from_location = k
                        break

            # Heuristic: try to move to the drawer area
            for k in positions.keys():
                if 'drawer' in k and 'area' in k:
                    to_location = k
                    break

            if from_location is None or to_location is None:
                found_locations = [k for k in positions.keys()]
                print("[Exploration] Could not clearly identify from/to locations. Candidates:", found_locations)
            else:
                print(f"[Exploration] Attempting to move robot from '{from_location}' to '{to_location}' using execute_go.")
                # Call the corresponding skill
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location,
                    to_location
                )
                print("[Exploration] execute_go action completed. Checking for missing predicates related to robot-at.")
                exploration_done = True

        except Exception as e:
            print("[Exploration] Exploration phase failed due to:", str(e))

        print("=== [Exploration] End ===")

        # -------------------
        # Task Plan Phase
        # -------------------
        # Based on the PDDL and feedback, you may need to:
        # - Move the robot to the relevant location (e.g., 'drawer-area')
        # - Manipulate objects: pick objects from floor, open/close drawers, place objects, etc.
        # Here, just as an example, perform a sequence with available skills.

        print("=== [Task Execution] Start ===")
        try:
            # Example: Move the robot to 'drawer-area' if not already there
            # Example keys: "drawer-area", "table-area", "initial_location" etc
            if not exploration_done:
                print("[Task] Attempting fallback move to 'drawer-area'")
                # This time, try known strings if not discovered in exploration
                from_location = "initial_location" if "initial_location" in positions else None
                to_location = "drawer-area" if "drawer-area" in positions else None
                if from_location and to_location:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location,
                        to_location
                    )
                else:
                    print("[Task] Could not determine valid from/to locations for execute_go")

            # Example handle manipulation: open a drawer (requires holding handle, drawer unlocked, at location etc.)
            # Use plausible keys from positions; the following is illustrative
            # You must ensure the correct object names based on your env and observation
            
            # 1. Pick up handle (assuming naming: 'handle', location: 'drawer-area')
            handle_name = None
            for k in positions.keys():
                if 'handle' in k:
                    handle_name = k
                    break

            drawer_name = None
            for k in positions.keys():
                if 'drawer' in k and 'handle' not in k and 'area' not in k:
                    drawer_name = k
                    break

            drawer_area = None
            for k in positions.keys():
                if 'drawer-area' in k:
                    drawer_area = k
                    break

            if handle_name and drawer_area:
                print(f"[Task] Attempting to pick up handle '{handle_name}' at '{drawer_area}'")
                try:
                    obs, reward, done = execute_pick(env, task, handle_name, drawer_area)
                except Exception as e:
                    print("[Task] Failed to execute_pick handle:", str(e))
            
            # 2. Pull handle to open drawer (requires holding handle, at drawer-area)
            if drawer_name and handle_name and drawer_area:
                print(f"[Task] Attempting to pull open drawer '{drawer_name}' using handle '{handle_name}' at '{drawer_area}'")
                try:
                    obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_area)
                except Exception as e:
                    print("[Task] Failed to execute_pull on drawer:", str(e))
            
            # 3. Example: Place an object in the open drawer
            # For full execution, identify a target object to place
            object_on_floor = None
            for k in positions.keys():
                if 'object' in k or 'item' in k or 'cube' in k:
                    object_on_floor = k
                    break

            if object_on_floor and drawer_name and drawer_area:
                print(f"[Task] Attempting to pick up object '{object_on_floor}' from floor at '{drawer_area}'")
                try:
                    obs, reward, done = execute_pick(env, task, object_on_floor, drawer_area)
                except Exception as e:
                    print("[Task] Failed to execute_pick on object:", str(e))

                print(f"[Task] Attempting to place '{object_on_floor}' into drawer '{drawer_name}' at '{drawer_area}'")
                try:
                    obs, reward, done = execute_place(env, task, object_on_floor, drawer_name, drawer_area)
                except Exception as e:
                    print("[Task] Failed to execute_place:", str(e))
            
            # 4. Optionally, push the drawer closed
            if drawer_name and drawer_area:
                print(f"[Task] Attempting to push drawer '{drawer_name}' closed at '{drawer_area}'")
                try:
                    obs, reward, done = execute_push(env, task, drawer_name, drawer_area)
                except Exception as e:
                    print("[Task] Failed to execute_push:", str(e))

            print("[Task Execution] plan completed.")

        except Exception as e:
            print("[Task Execution] Task phase failed due to:", str(e))

        print("=== [Task Execution] End ===")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
