# run_skeleton_task.py (Completed for Diagnostic/Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation—including missing predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer3': ..., ...}
        positions = get_object_positions()

        # === Diagnostic Exploration: Find Missing Predicate (e.g., drawer-open drawer3) ===
        #
        # Problem: Feedback indicates (drawer-open drawer3) -- need to explore and verify drawer3's state
        #
        # Plan:
        #   1. Locate drawer3 (and its handle/object).
        #   2. Use available exploration skills (execute_pull, execute_go, etc.) if needed.
        #   3. Try to open, push, or otherwise interact to induce/observe the drawer-open property.
        #   4. Log/print relevant states for debugging.

        # Identify available objects and drawers in positions
        drawer_name = 'drawer3'
        handle_name = None
        for obj_name in positions:
            # For this domain, handles might be named or associated with drawers
            if 'handle' in obj_name and drawer_name in obj_name:
                handle_name = obj_name
                break

        if drawer_name not in positions:
            print(f"[Error] {drawer_name} not present in positions: {list(positions.keys())}")
            return
        if handle_name is None:
            # Try generic approach - look for any 'handle' and hope it's mapped
            for obj_name in positions:
                if 'handle' in obj_name:
                    handle_name = obj_name
                    break
        if handle_name is None:
            print(f"[Warning] No handle found for {drawer_name}; some actions may not be possible.")
        
        robot_location = None
        # Try to find robot's current location from the initial observation
        for name, pos in positions.items():
            if 'robot' in name or 'Robot' in name:
                robot_location = name
                break
        if robot_location is None:
            # Fallback; may depend on env/task internals
            robot_location = 'base'
        # Target location is where the drawer is
        drawer_location = drawer_name

        # === PHASE 1: Move to drawer3's location ===
        try:
            print(f"[Exploration] Moving robot to {drawer_location} to investigate state of {drawer_name}")
            obs, _, done = execute_go(env, task, robot_location, drawer_location)
            robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # === PHASE 2: Attempt to Open or Interact with Drawer to Test 'drawer-open' Predicate ===
        #
        # Use skill primitives to check drawer state transitions;
        # Particularly, execute_pull tries to open it (requires: holding the handle, unlocked, and closed)
        #

        # First, try to pick the handle if present
        picked_handle = False
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick up handle {handle_name} for {drawer_name}")
                obs, _, done = execute_pick(env, task, handle_name, drawer_location)
                picked_handle = True
            except Exception as e:
                print(f"[Exploration] Could not pick up handle {handle_name}: {e}")

        # Now, try to pull the drawer open if handle is picked
        pulled = False
        if picked_handle and handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pull (open) {drawer_name} using handle {handle_name}")
                obs, _, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                pulled = True
            except Exception as e:
                print(f"[Exploration] Could not pull to open {drawer_name}: {e}")

        # If we failed to pick or pull, or handle is missing, try to push (maybe already open and needs to close first)
        if not pulled:
            try:
                print(f"[Exploration] Attempting to push {drawer_name} (try to close if open before another open attempt)")
                obs, _, done = execute_push(env, task, drawer_name, drawer_location)
                # Now attempt pull again
                if handle_name is not None:
                    print("[Exploration] Retrying: pick handle then pull after closing drawer.")
                    obs, _, done = execute_pick(env, task, handle_name, drawer_location)
                    obs, _, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                    pulled = True
            except Exception as e:
                print(f"[Exploration] Error during execute_push/repull: {e}")

        # Diagnostic: check observable state if possible
        try:
            obs_now = task.get_observation()
            print(f"[Exploration] Post-actions observation: {obs_now}")
        except Exception as e:
            print(f"[Exploration] Could not retrieve observation: {e}")

        print(f"[Exploration] Exploration phase complete for missing predicate on {drawer_name}.")

        # === Continue possibly with further skill checks if necessary ===

        # This is where you would add domain-specific plan logic after discovering the actual predicate state and available actions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()