# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Predefined skill primitives: do not redeclare

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Relevant Info ===
        positions = get_object_positions()

        # --- EXPLORATION PHASE ---
        # Based on feedback, need to check if predicate (drawer-open drawer3) is missing,
        # i.e., whether the drawer3 is open or closed, and possibly identify its openable condition.
        #
        # Use the available skills and environment info for this phase.

        # Assumed convention: drawer object names, handle objects, and locations are reported in 'positions'
        # No skill functions are redefined - assume their API matches skill_code definitions.

        # Replace these with actual IDs in your env if needed
        drawer3_name = 'drawer3'
        probable_handle = None  # Will be set after handle discovery

        # Try to detect handle object for drawer3
        for obj, attr in positions.items():
            if 'handle-of' in attr and attr['handle-of'] == drawer3_name:
                probable_handle = obj
                break

        # Use fallback if not directly given (example: "handle3" or similar)
        if probable_handle is None:
            for obj in positions.keys():
                if f'handle' in obj and '3' in obj:
                    probable_handle = obj
                    break

        if probable_handle is None:
            print("[Exploration] No handle found for drawer3! Exploration aborted.")
            return

        print(f"[Exploration] Found probable handle for {drawer3_name}: {probable_handle}")

        # Step 1: Go to handle's location
        robot_current_location = None
        for k, v in positions.items():
            if k == 'robot' or 'robot' in k:
                robot_current_location = v['location']
        if robot_current_location is None:
            print("[Exploration] Could not determine robot's current location")
            return

        handle_location = None
        if 'location' in positions[probable_handle]:
            handle_location = positions[probable_handle]['location']
        else:
            # Fallback: check parent drawer's location or similar
            if drawer3_name in positions and 'location' in positions[drawer3_name]:
                handle_location = positions[drawer3_name]['location']

        if handle_location is None:
            print("[Exploration] Could not determine handle location for exploration")
            return

        try:
            # Move to handle location
            print(f"[Exploration] Moving to handle location: {handle_location}")
            obs, reward, done, info = execute_go(
                env, task,
                from_location=robot_current_location,
                to_location=handle_location
            )
        except Exception as e:
            print("[Exploration] execute_go failed:", e)
            return

        # Step 2: Pick the handle (only if free)
        try:
            print(f"[Exploration] Attempting to pick up handle: {probable_handle}")
            obs, reward, done, info = execute_pick(
                env, task,
                object_name=probable_handle,
                location=handle_location
            )
        except Exception as e:
            print("[Exploration] execute_pick failed (possibly already holding or hand not empty):", e)
            # Try to free hand if needed
            try:
                obs, reward, done, info = execute_gripper(env, task)
            except Exception:
                pass
            # Try picking again
            try:
                obs, reward, done, info = execute_pick(
                    env, task,
                    object_name=probable_handle,
                    location=handle_location
                )
            except Exception as e:
                print("[Exploration] execute_pick retry failed:", e)
                return

        # Step 3: Try to open the drawer using the pull skill (explore predicates)
        try:
            print(f"[Exploration] Attempting to pull and open drawer: {drawer3_name} via handle: {probable_handle}")
            obs, reward, done, info = execute_pull(
                env, task,
                drawer_name=drawer3_name,
                handle_name=probable_handle,
                location=handle_location
            )
            print("[Exploration] execute_pull succeeded.")
        except Exception as e:
            print("[Exploration] execute_pull failed -- likely due to missing precondition (e.g., drawer locked or not at right location):", e)
            # Here, in real exploration, could attempt to unlock the drawer if a skill existed

        # Step 4: Check the state of the drawer (is the missing predicate now true?)
        updated_positions = get_object_positions()
        drawer_status = None
        if drawer3_name in updated_positions and 'status' in updated_positions[drawer3_name]:
            drawer_status = updated_positions[drawer3_name]['status']
        else:
            # Fallback: logical check
            if drawer3_name in updated_positions and 'open' in updated_positions[drawer3_name]:
                drawer_status = 'open' if updated_positions[drawer3_name]['open'] else 'closed'
        print(f"[Exploration] Drawer3 status after pull attempt: {drawer_status}")

        # If needed, explore further (e.g., try to unlock), but no unlock skill available in predefined set.

        # The exploration phase finished. Normally, additional code would use discovered info
        # (e.g., to compose a plan involving execute_place, execute_push, etc.).

        # --- END EXPLORATION ---

        print("[Task] Exploration phase complete. Proceed with downstream task steps as needed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
