# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing skill implementations

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running exploration to determine missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Types ===
        positions = get_object_positions()

        # Assume object_positions provides at least the following keys:
        #   - object locations (e.g. {'obj1': (x,y,z), ...})
        #   - drawer names and handle names, if present
        #   - a mapping such as positions['drawers'], positions['handles'], positions['objects'], positions['locations']

        # For the purpose of generic exploration, we search for drawers and check their open/closed state.
        try:
            drawers = positions.get('drawers', [])
            handles = positions.get('handles', [])
            robot_location = positions.get('robot', positions.get('robot-location', None))
            objects = positions.get('objects', [])
            locations = positions.get('locations', [])
        except Exception as e:
            print("[Error] Problem extracting keys from object positions:", str(e))
            drawers, handles, robot_location, objects, locations = [], [], None, [], []

        # === EXPLORATION LOGIC ===

        # 1. Try basic predicate exploration by attempting known actions and checking their outcomes.
        #    Here we try to pull on each drawer to see if some unknown precondition is blocking us (e.g., drawer-closed).

        explored = False
        for drawer in drawers:
            print(f"[Exploration] Checking drawer: {drawer}")
            drawer_loc = None
            try:
                drawer_info = positions.get(drawer, {})
                if isinstance(drawer_info, dict) and 'location' in drawer_info:
                    drawer_loc = drawer_info['location']
                elif isinstance(drawer_info, tuple):
                    drawer_loc = drawer_info
                else:
                    drawer_loc = None
            except Exception as e:
                print(f"  [Exploration] Could not get location for {drawer}: {e}")
                continue

            # Find the handle associated with this drawer
            drawer_handle = None
            for handle in handles:
                handle_info = positions.get(handle, {})
                # Usually there will be a mapping or known convention for which handle matches which drawer
                # For this example, assume positions['handle_of'][handle] = drawer
                # Otherwise, use naming conventions
                if 'handle_of' in positions:
                    if positions['handle_of'].get(handle, None) == drawer:
                        drawer_handle = handle
                        break
                if handle.startswith(drawer) or drawer in handle:
                    drawer_handle = handle
                    break

            if not drawer_handle:
                print(f"  [Exploration] No handle found for drawer {drawer}. Skipping.")
                continue

            print(f"  [Exploration] Found handle {drawer_handle} for drawer {drawer} at {drawer_loc}")

            # 2. Try to move robot to the drawer's location
            if robot_location and drawer_loc and robot_location != drawer_loc:
                try:
                    print(f"  [Exploration] Moving robot from {robot_location} to {drawer_loc}")
                    obs, reward, done, info = execute_go(
                        env,
                        task,
                        from_location=robot_location,
                        to_location=drawer_loc
                    )
                    robot_location = drawer_loc
                except Exception as e:
                    print(f"    [Error] execute_go failed: {e}")

            # 3. Try to pick the handle (precondition of execute_pull is holding(handle), handle-of, drawer-closed, drawer-unlocked, robot-at)
            try:
                print(f"  [Exploration] Attempting to pick handle {drawer_handle}")
                obs, reward, done, info = execute_pick(
                    env,
                    task,
                    object_name=drawer_handle,
                    location=drawer_loc
                )
                # Expectation: After successful pick, robot is holding the handle
            except Exception as e:
                print(f"    [Error] execute_pick failed: {e}")

            # 4. Try to pull the drawer (should fail if missing a predicate such as (drawer-closed X) etc)
            try:
                print(f"  [Exploration] Attempting to pull (open) drawer {drawer}")
                obs, reward, done, info = execute_pull(
                    env,
                    task,
                    drawer_name=drawer,
                    handle_name=drawer_handle,
                    location=drawer_loc
                )
                explored = True
                print(f"    [Exploration] Pull action executed. TEST for missing predicates in logs.")
            except Exception as e:
                # Log the error message to help identify which predicate is missing
                print(f"    [Exploration] execute_pull failed. Review error/logs to identify missing predicate. Error: {e}")
                # e.g., error string might reveal missing 'drawer-closed' or similar

            # 5. Done with one drawer for this exploration
            break

        # === If further feedback required, can attempt additional exploration ===
        # e.g., attempt to push closed drawers, place objects, etc.

        if not explored:
            print("[Exploration] No suitable drawer found or could not complete exploration steps.")

        # Observation: After attempting the above, check the outputs/errors/logs to see
        # if the 'drawer-closed' predicate is referenced as a missing precondition.
        print("[Result] If an error mentioning 'drawer-closed' appears, the predicate is missing in the initial state.")

        # (You can expand this by iterating over more objects, drawers, and locations for complete environment probing.)

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()