# run_skeleton_task.py (COMPLETE EXECUTABLE CODE)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # You don't need to redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running exploration and any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE FOR MISSING PREDICATE (e.g., robot-at drawer-area) ===
        # The missing predicate from feedback: (robot-at drawer-area)
        # We need to check if the robot can move to 'drawer-area' using the available skills

        # Example: Let's try to execute_go to 'drawer-area' (if 'drawer-area' exists in positions)
        current_robot_location = None
        for k in positions.keys():
            if "robot" in k:
                current_robot_location = positions[k]
                break

        # If not found by key, try a fallback (assume initial location)
        # For this example, let's just pick the first location as a placeholder
        if current_robot_location is None:
            current_robot_location = list(positions.values())[0]

        # Try to find the 'drawer-area' location in the environment
        drawer_area_location = None
        for obj_name, pos in positions.items():
            if "drawer-area" in obj_name or "drawer" in obj_name:
                drawer_area_location = pos
                break

        if drawer_area_location is None:
            print("[Exploration] No 'drawer-area' found in current object positions. Skipping exploration step.")
        else:
            print("[Exploration] Attempting to move robot to 'drawer-area' to test predicate (robot-at drawer-area)")
            try:
                # Use the skill execute_go to move the robot from current location to drawer-area
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_robot_location, 
                    to_location=drawer_area_location,
                    max_steps=100
                )
                print("[Exploration] Robot moved to 'drawer-area'. This should set (robot-at drawer-area).")
            except Exception as e:
                print(f"[Exploration] Failed to move to drawer-area: {e}")

        # === Main Task Plan Execution (Example Sequence) ===
        # Now perform a sequence of skills available to the task.
        # The actual plan would depend on the task and environment, but here's a representative sample:
        # This will be a generic demonstration covering available skills.

        # Example: Try to pick and place an object in the drawer

        # Step 1: Locate an object on the floor
        object_to_pick = None
        object_position = None
        for name, pos in positions.items():
            # Assume naming convention: objects on floor contain "obj" or "object"
            if "obj" in name or "object" in name:
                object_to_pick = name
                object_position = pos
                break

        if object_to_pick is not None and object_position is not None:
            print(f"[Task] Attempting to pick up {object_to_pick} at {object_position}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_to_pick,
                    location=object_position,
                    max_steps=100
                )
                if done:
                    print(f"[Task] Picked up {object_to_pick}.")
            except Exception as e:
                print(f"[Task] Failed to pick up {object_to_pick}: {e}")
        else:
            print("[Task] No object found to pick up.")

        # Step 2: Find a drawer object and its handle
        drawer_obj = None
        drawer_pos = None
        handle_obj = None
        handle_pos = None

        for name, pos in positions.items():
            if "drawer" in name and "handle" not in name:
                drawer_obj = name
                drawer_pos = pos
            if "handle" in name:
                handle_obj = name
                handle_pos = pos

        # Step 3: Try to pull (open) the drawer using the handle if we have both
        if drawer_obj is not None and handle_obj is not None and drawer_pos is not None and handle_pos is not None:
            print(f"[Task] Attempting to pull open {drawer_obj} using {handle_obj}.")
            try:
                # First, pick the handle if not holding it
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=handle_obj,
                    location=handle_pos,
                    max_steps=100
                )
                if done:
                    print(f"[Task] Picked up {handle_obj}.")
                # Now, pull/open the drawer
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_obj,
                    handle_name=handle_obj,
                    location=drawer_pos,
                    max_steps=100
                )
                if done:
                    print(f"[Task] Drawer {drawer_obj} opened.")
            except Exception as e:
                print(f"[Task] Failed to open drawer {drawer_obj}: {e}")
        else:
            print("[Task] Drawer or handle not found; cannot perform pull.")

        # Step 4: Place the object in the drawer (if object and drawer available)
        if object_to_pick is not None and drawer_obj is not None and drawer_pos is not None:
            print(f"[Task] Attempting to place {object_to_pick} in {drawer_obj}.")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj_name=object_to_pick,
                    drawer_name=drawer_obj,
                    location=drawer_pos,
                    max_steps=100
                )
                if done:
                    print(f"[Task] Placed {object_to_pick} in {drawer_obj}.")
            except Exception as e:
                print(f"[Task] Failed to place {object_to_pick} in {drawer_obj}: {e}")
        else:
            print("[Task] Cannot place object; missing drawer or object.")

        # (Optionally continue with more steps as per oracle plan...)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
