# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Record task steps and observations if needed
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # EXPLORE: Feedback is that '(robot-at drawer-area)' is missing.
        # Exploration phase to investigate which predicates are missing, by navigating to 'drawer-area'

        # Attempt to move robot to 'drawer-area' or the location believed to be 'drawer-area'
        # Note: As we do not know the exact robot or location object names from (:objects ...), we infer from context

        # Attempt to identify possible location keys for the drawer
        location_candidates = []
        for key in positions.keys():
            if "drawer" in key or "Drawer" in key:
                location_candidates.append(key)
        if len(location_candidates) == 0:
            # Fallback: try common location keys
            for key in positions.keys():
                if "area" in key or "position" in key or "zone" in key:
                    location_candidates.append(key)
        if len(location_candidates) == 0:
            # Fallback to any available location
            location_candidates = [k for k, v in positions.items() if isinstance(v, (list, tuple, np.ndarray))]

        if len(location_candidates) > 0:
            drawer_location = location_candidates[0]
        else:
            print("[Exploration] Warning: Could not find a plausible drawer-area location in object positions.")
            return

        # Identify current robot position
        robot_loc = None
        for k, v in positions.items():
            if "robot" in k.lower():
                robot_loc = k
                break
        if robot_loc is None:
            # If not known, pick a source location different from the drawer (if possible)
            for k in positions.keys():
                if k != drawer_location:
                    robot_loc = k
                    break
        if robot_loc is None:
            # Default to first found location
            robot_loc = drawer_location

        print(f"[Exploration] Robot will attempt to move from '{robot_loc}' to '{drawer_location}'.")

        # Use the available skill to go/move to the drawer area
        goal_reached = False
        try:
            # Skill: execute_go or execute_go_identify may exist; prefer execute_go for primitive nav
            if 'execute_go' in globals():
                obs, reward, done = execute_go(env, task, robot_loc, drawer_location)
            else:
                print("[Exploration] ERROR: No execute_go skill available.")
                return
            goal_reached = True
            print(f"[Exploration] Robot arrived at {drawer_location}. Feedback: predicate '(robot-at {drawer_location})' should now be true.")
        except Exception as ex:
            print(f"[Exploration] Exception during navigation: {ex}")

        # If successful, attempt to trigger feedback detection (exploration of missing predicate)
        if goal_reached:
            # Optionally, try to 'identify' objects or states at the new location using available skills
            if 'execute_go_identify' in globals():
                try:
                    obs, reward, done = execute_go_identify(env, task, robot_loc, drawer_location)
                    print(f"[Exploration] Environment explored via execute_go_identify at {drawer_location}.")
                except Exception as ex:
                    print(f"[Exploration] Exception with execute_go_identify: {ex}")
            else:
                print(f"[Exploration] No execute_go_identify skill, relying on execute_go result.")

        # [Your PLAN CODE HERE: Insert further task steps using available skills as needed.]
        # For this exploration assignment, we terminate after demonstrating the missing predicate logic.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()