# run_skeleton_task.py (Fully Generic Skeleton - Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example expected keys: 'drawer1', 'robot', 'handle1', etc.

        # --- Exploration Phase: Find Missing Predicate via Drawer Open Status ---
        # Based on feedback that (drawer-open drawer1) is relevant, we want to check if the state can be set, i.e.,
        # explore the agent's ability to open the drawer and observe the effect.

        # If the environment provides names, we expect something like:
        # positions['drawer1'], positions['handle1'], positions['robot'], etc.

        drawer_name = 'drawer1'
        handle_name = 'handle1'
        location_name = 'location_drawer1'
        robot_current_location = None

        # Use try/except in case keys are not present (robust to missing info)
        try:
            drawer_pos = positions[drawer_name]
        except Exception:
            print(f"[Warning] '{drawer_name}' not found in positions map.")
            drawer_pos = None
        
        try:
            handle_pos = positions[handle_name]
        except Exception:
            handle_pos = None
        
        try:
            robot_pos = positions['robot']
        except Exception:
            robot_pos = None

        # Identify possible location name for robot and drawer
        if 'location_drawer1' in positions:
            location_name = 'location_drawer1'
        elif 'drawer1_location' in positions:
            location_name = 'drawer1_location'
        elif 'drawer1' in positions:
            # Could use the same key for location
            location_name = 'drawer1'
        else:
            location_name = None

        # Step 1: Move robot to drawer
        try:
            # Determine current and target location
            if robot_pos is not None and drawer_pos is not None:
                # Simulate movement from current robot position to drawer location
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_pos,
                    to_location=drawer_pos,
                    max_steps=100,
                    timeout=10.0
                )
                robot_current_location = drawer_pos
            elif location_name is not None:
                # Use symbolic location names if positions unavailable
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='robot_home',
                    to_location=location_name,
                    max_steps=100,
                    timeout=10.0
                )
                robot_current_location = location_name
            else:
                print("[Warning] No valid location for robot and/or drawer available. Skipping move.")
        except Exception as e:
            print(f"[Error] execute_go failed: {str(e)}")

        # Step 2: Pick up handle (required for pulling/opening drawer)
        try:
            if handle_name in positions:
                targ_handle_pos = positions[handle_name]
            else:
                targ_handle_pos = drawer_pos  # fallback
            obs, reward, done = execute_pick(
                env,
                task,
                target_object=handle_name,
                location=robot_current_location if robot_current_location else location_name,
                max_steps=100,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_pick on {handle_name}: {str(e)}")

        # Step 3: Try to open the drawer (pull action)
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                handle=handle_name,
                location=robot_current_location if robot_current_location else location_name,
                max_steps=100,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_pull on {drawer_name}: {str(e)}")

        # Step 4: Check if the missing predicate (drawer-open) is true by observation
        # (Assume you can call task.get_observation() to get the latest belief/state)
        final_obs = None
        try:
            final_obs = task.get_observation()
            if 'drawer-open' in str(final_obs) or "(drawer-open {})".format(drawer_name) in str(final_obs):
                print(f"[Exploration] Successfully inferred predicate: (drawer-open {drawer_name}) is now true")
            else:
                print(f"[Exploration] Could not confirm (drawer-open {drawer_name}) in observation. Observed: {final_obs}")
        except Exception as e:
            print(f"[Error] While checking for (drawer-open {drawer_name}): {str(e)}")

        # --- End of Exploration ---
        # If running an oracle plan: insert additional skills as needed, e.g., place or further manipulation

        # Example: Place an object in the open drawer if plan requires
        # try:
        #     obs, reward, done = execute_place(
        #         env,
        #         task,
        #         object_name='object1',
        #         drawer=drawer_name,
        #         location=robot_current_location if robot_current_location else location_name,
        #         max_steps=100,
        #         timeout=10.0
        #     )
        # except Exception as e:
        #     print(f"[Error] execute_place on {drawer_name}: {str(e)}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()