# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    missing_predicates = []
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration: Find Missing Predicate "robot-at drawer-area" ===
        # According to feedback, (robot-at drawer-area) is missing. We need to determine if
        # our environment encodes 'robot-at' predicates or if this is a naming/config mismatch.

        # Let's attempt to explore all positions/locations declared in the environment.
        # We will try to move to 'drawer-area' and check if the predicate appears in observations.

        # First, get candidate locations from the positions dict
        candidate_locations = [k for k in positions if 'area' in k or 'drawer' in k or 'location' in k]
        
        found_robot_at_drawer_area = False
        for loc in candidate_locations:
            try:
                # Try to move robot to candidate location using available skill
                # The canonical movement skill is execute_go (from ?from ?to)
                # We'll need to know current location. Suppose we can get it either from initial obs or by convention.
                # If not available, try all as from-loc to to-loc
                from_locs = [k for k in candidate_locations if k != loc]
                for from_loc in from_locs:
                    try:
                        print(f"[Exploration] Trying to move from {from_loc} to {loc} using execute_go")
                        obs_, reward_, done_ = execute_go(env, task, from_loc, loc)
                        curr_obs = task.get_observation()
                        # Check if predicates about robot-at are in obs or description
                        if "robot-at" in str(curr_obs) and "drawer-area" in str(curr_obs):
                            print("[Exploration] Found missing predicate: (robot-at drawer-area)")
                            found_robot_at_drawer_area = True
                            break
                    except Exception as move_exc:
                        print(f"[Exploration] Movement from {from_loc} to {loc} failed: {move_exc}")
                if found_robot_at_drawer_area:
                    break
            except Exception as exc:
                print(f"[Exploration] Could not try moving to {loc}: {exc}")

        if not found_robot_at_drawer_area:
            missing_predicates.append("robot-at drawer-area")

        # === Plan Execution Example ===
        # Below is a generic approach to achieving a goal after exploration.
        # For your domain (drawer, pick, place, etc.), present a sequence that uses only the supplied skills.

        try:
            # This is a sample canonical plan for a "pick and place into drawer" task

            # -- 1. Move to drawer-area (if required)
            robot_pos = None
            if found_robot_at_drawer_area:
                robot_pos = "drawer-area"
            else:
                # Attempt to get a fallback location
                robot_pos = candidate_locations[0] if candidate_locations else None

            # -- 2. Identify objects on floor to pick
            objects_on_floor = [k for k, v in positions.items() if 'object' in k or 'item' in k]
            target_obj = objects_on_floor[0] if objects_on_floor else None

            # 3. Picking up an object on the floor (assume pick at current robot position)
            #    Assumes robot-free, hand-empty, and robot-at
            if target_obj and robot_pos:
                print(f"[Task] Picking up {target_obj} at location {robot_pos}")
                try:
                    obs, reward, done = execute_pick(env, task, target_obj, robot_pos)
                except Exception as pick_exc:
                    print(f"[Task] execute_pick failed: {pick_exc}")

            # 4. Open Drawer -- requires handle
            # Find a handle, and use execute_pull to open drawer
            handles = [k for k in positions if 'handle' in k]
            drawers = [k for k in positions if 'drawer' in k]
            if handles and drawers and robot_pos:
                handle = handles[0]
                drawer = drawers[0]
                try:
                    # Must first pick the handle
                    print(f"[Task] Picking up handle {handle} at {robot_pos}")
                    obs, reward, done = execute_pick(env, task, handle, robot_pos)
                    print(f"[Task] Pulling drawer {drawer} with handle {handle} at {robot_pos}")
                    obs, reward, done = execute_pull(env, task, drawer, handle, robot_pos)
                except Exception as pull_exc:
                    print(f"[Task] execute_pull failed: {pull_exc}")

            # 5. Place object into open drawer
            if target_obj and drawers and robot_pos:
                drawer = drawers[0]
                try:
                    print(f"[Task] Placing {target_obj} into drawer {drawer} at {robot_pos}")
                    obs, reward, done = execute_place(env, task, target_obj, drawer, robot_pos)
                except Exception as place_exc:
                    print(f"[Task] execute_place failed: {place_exc}")

            # 6. Close drawer
            if drawers and robot_pos:
                drawer = drawers[0]
                try:
                    print(f"[Task] Pushing drawer {drawer} closed at {robot_pos}")
                    obs, reward, done = execute_push(env, task, drawer, robot_pos)
                except Exception as push_exc:
                    print(f"[Task] execute_push failed: {push_exc}")

        except Exception as task_exc:
            print(f"[Task] Task execution failed: {task_exc}")

        # === Exploration Report ===
        if missing_predicates:
            print(f"[Exploration] Missing predicates: {missing_predicates}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
