# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Assume all required skill primitives are imported

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()
        
        # For this task, we proceed to perform EXPLORATION STEPS to discover what predicates are missing.
        # Example feedback indicates possible missing knowledge like (robot-at drawer-area)
        # We'll use exploration skills as per the available skills and feedback.

        # ----- EXPLORATION PHASE START -----
        # Let's try to navigate to the 'drawer-area' to ensure (robot-at drawer-area) holds,
        # and perform sweeps/gripper actions as required to trigger predicates or discoverable knowledge.

        robot_location = None
        try:
            # In some envs, the task may provide robot's current location. Otherwise, use a default key.
            robot_location = positions.get('robot', {}).get('location', None)
        except Exception:
            pass

        # For safety, default location
        if robot_location is None:
            robot_location = 'start_area'  # Or the corresponding string for the initial robot location

        # Key location to approach based on feedback
        target_location = 'drawer-area'

        # Attempt to move robot to the drawer area if not already there
        try:
            # Use execute_go as per primitive_skills_static_lock_v2 domain
            obs, reward, done = execute_go(
                env, 
                task, 
                from_location=robot_location, 
                to_location=target_location, 
                max_steps=50
            )
            print(f"[Exploration] Robot moved from {robot_location} to {target_location}.")
            robot_location = target_location
        except Exception as e:
            print(f"[Exploration] Failed to move robot: {e}")

        # Now that robot is at drawer-area, check if any predicates can be triggered.
        # Optionally, try exploration skills such as 'execute_sweep' or 'execute_gripper', which may
        # further reveal the missing predicate.

        # Try sweeping the area to "sense" or "identify" objects
        try:
            # Try on a list of all objects at drawer-area
            objects_in_area = []
            for obj_name, obj_pos in positions.items():
                # If position or area indicates at drawer-area, append
                if isinstance(obj_pos, dict) and obj_pos.get('area', None) == 'drawer-area':
                    objects_in_area.append(obj_name)
            # If area information isn't in positions, treat all known objects as candidates
            if not objects_in_area:
                objects_in_area = [obj for obj in positions if obj != 'robot']
            
            for obj in objects_in_area:
                try:
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        o=obj,
                        p=target_location,
                        max_steps=25
                    )
                    print(f"[Exploration] Swept object '{obj}' at '{target_location}'")
                except Exception as ex:
                    print(f"[Exploration] Sweep failed for '{obj}': {ex}")

        except Exception as e:
            print(f"[Exploration] Sweep phase: {e}")

        # Optionally, try toggling gripper to trigger robot-free or hand-empty related knowledge
        try:
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=20
            )
            print(f"[Exploration] Executed gripper action at '{target_location}'")
        except Exception as e:
            print(f"[Exploration] Gripper action failed: {e}")

        # ----- EXPLORATION PHASE END -----

        print("[Exploration] Done. If (robot-at drawer-area) is present, we've verified location knowledge.")

        # Proceed with oracle plan here (currently not provided in this code-gen, since it's skeleton + exploration task)
        # In realistic usage, after exploration you'd invoke primitives like execute_pick, execute_place,
        # execute_push, execute_pull, etc., according to the domain's requirements and oracle plan steps.

        # Example (commented):
        # obs, reward, done = execute_pick(env, task, o='target_obj', p=target_location, max_steps=100)

        # If desired, check if exploration yielded new observations
        current_obs = task.get_observation()
        print("[Current Observation]", current_obs)
        # ... Assumed that downstream logic will use these observations to update plan

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()