# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # pre-defined primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = 'object_1': (0,0,0), 'object_2': (1,1,1), ... (dictionary format)
        positions = get_object_positions()

        print("[Exploration] Checking predicates for robot location...")

        # --- Begin Exploration Phase to find missing predicate as described from feedback ---

        # Suppose the task needs to check for 'robot-at table' predicate as given by feedback
        table_position = None

        # Try to find a location that could represent the 'table'
        for name, pos in positions.items():
            if "table" in name.lower():
                table_position = pos
                table_key = name
                break

        # If not in positions, try a generic approach for possible location keys
        if table_position is None:
            for name, pos in positions.items():
                if "loc" in name.lower() or "location" in name.lower():
                    table_position = pos
                    table_key = name
                    break

        if table_position is None:
            print("[Warning] Could not find any location corresponding to 'table' in object positions.")
            # Can't proceed with 'robot-at table' exploration
        else:
            print(f"[Exploration] Found table at '{table_key}' at position {table_position}")

            try:
                # Use the available skill to go/move to the table location
                # We suppose the robot has to go to the table, so we need its current location;
                # For that, see which location predicates are supplied
                # As we don't have actual location keys, let's go from current position (guess first key) to table

                current_location_key = None
                for name, pos in positions.items():
                    if "robot" in name.lower() or "start" in name.lower():
                        current_location_key = name
                        break
                if current_location_key is None:
                    # Just pick the first location key that is not the table
                    for name, pos in positions.items():
                        if name != table_key:
                            current_location_key = name
                            break

                if current_location_key:
                    print(f"[Exploration] Robot moving from '{current_location_key}' to '{table_key}'")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=current_location_key,
                        to_location=table_key,
                        current_pos=positions[current_location_key],
                        target_pos=table_position,
                        approach_distance=0.10,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print("[Exploration] robot-at table predicate explored")
                else:
                    print("[Warning] Could not determine robot's current location, skipping go-to-table exploration.")

            except Exception as e:
                print(f"[Exception] During exploration to table: {e}")

        # --- Placeholder: If more predicates are to be explored, add more skill invocations here ---

        print("[Exploration] Exploration phase for predicate completion finished.")

        # === Task Plan Execution ===
        # At this point, you would continue with your plan or episode logic using only given primitive skills.
        # This code structure allows you to add steps based on the environment and domain logic, utilizing
        # execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper as needed.
        #
        # For example:
        #
        # if need_to_pick_object:
        #     obs, reward, done = execute_pick(env, task, ...)
        #
        # if need_to_place_object:
        #     obs, reward, done = execute_place(env, task, ...)
        #

        # Note: Exploration code above is abstract/generic, to be concretized step-by-step for your specific oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
