# run_skeleton_task.py (Filled with Exploration Phase and Skill Integration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only external, do not add/modify any skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''

    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === ===================================================================
        #  Exploration Phase: Detect Missing Predicates (Lock Knowledge etc.)
        # =======================================================================
        #
        # Since the feedback and exploration domain suggest that missing predicates 
        # might relate to whether an object/drawer is "lock-known", the code should 
        # implement an exploration phase: Try to interact with possibly unrecognized
        # drawer states using the available skills, and observe any errors or return
        # values that would indicate missing preconditions or predicate knowledge.

        # --- Start Exploration ---
        exploration_done = False
        predicate_missing = None

        # Gather object/drawer/location info (replace these by your actual object/drawer names as needed)
        # We'll try to find a drawer and its handle.
        drawer_name = None
        handle_name = None
        drawer_location = None
        handle_location = None
        found_drawer = False
        found_handle = False

        for obj_name, obj_info in positions.items():
            if 'drawer' in obj_name and not found_drawer:
                drawer_name = obj_name
                if isinstance(obj_info, dict) and 'location' in obj_info:
                    drawer_location = obj_info['location']
                else:
                    drawer_location = obj_info
                found_drawer = True
            if 'handle' in obj_name and not found_handle:
                handle_name = obj_name
                if isinstance(obj_info, dict) and 'location' in obj_info:
                    handle_location = obj_info['location']
                else:
                    handle_location = obj_info
                found_handle = True

        # If not found, try fallback  
        if not drawer_location or not drawer_name:
            drawer_name, drawer_location = next(((k, v['location']) for k, v in positions.items() if 'drawer' in k and 'location' in v), (None, None))
        if not handle_location or not handle_name:
            handle_name, handle_location = next(((k, v['location']) for k, v in positions.items() if 'handle' in k and 'location' in v), (None, None))

        # Also pick some 'object' on the floor (ball, eraser, etc.)
        object_name = None
        object_location = None
        for obj_name, obj_info in positions.items():
            if 'handle' not in obj_name and 'drawer' not in obj_name:
                object_name = obj_name
                if isinstance(obj_info, dict) and 'location' in obj_info:
                    object_location = obj_info['location']
                else:
                    object_location = obj_info
                break

        # Find current robot location if possible
        current_robot_pos = None
        if 'robot' in positions:
            robot_entry = positions['robot']
            if isinstance(robot_entry, dict) and 'location' in robot_entry:
                current_robot_pos = robot_entry['location']
            elif isinstance(robot_entry, (tuple, list)):
                current_robot_pos = robot_entry
        if current_robot_pos is None:
            # Fallback: select any location in positions (keys often look like 'location_X')
            for k, v in positions.items():
                if 'location' in k.lower():
                    current_robot_pos = v if isinstance(v, (tuple, list)) else (v['location'] if 'location' in v else None)
                    break

        # If there is something like a list of location names, just pick two
        location_names = [k for k in positions.keys() if 'location' in k.lower() or 'loc' in k.lower()]
        if not location_names and current_robot_pos is not None:
            location_names = [current_robot_pos]
        elif current_robot_pos is not None:
            if current_robot_pos not in location_names:
                location_names.append(current_robot_pos)

        # If we are missing crucial names, abort the test
        if drawer_name is None or handle_name is None or current_robot_pos is None:
            print("[Exploration] Could not identify necessary drawer/handle/robot for exploration; skipping predicate check.")
            exploration_done = True

        # Try the exploration by invoking primitive actions and watch for missing conditions/errors
        # 1. Go to the drawer location if not already there
        if not exploration_done:
            try:
                if current_robot_pos != drawer_location:
                    print(f"[Exploration] execute_go: {current_robot_pos} → {drawer_location}")
                    # Use the name/ID (not position tuple) for the env, as skills typically expect string location identifiers
                    obs, reward, done = execute_go(env, task, current_robot_pos, drawer_location)
                    current_robot_pos = drawer_location
            except Exception as e:
                print(f"[Exploration] Exception while moving to drawer: {e}")

            # 2. Try picking up the handle (simulate robot preparing to pull drawer)
            try:
                print(f"[Exploration] execute_pick: {handle_name} at {drawer_location}")
                obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
            except Exception as e:
                err_msg = str(e)
                print(f"[Exploration] Exception on execute_pick: {err_msg}")
                if 'predicate' in err_msg.lower():
                    predicate_missing = err_msg.strip()
                    exploration_done = True

            # 3. Try pulling the drawer with the handle
            if not exploration_done:
                try:
                    print(f"[Exploration] execute_pull: Drawer={drawer_name} Handle={handle_name} at {drawer_location}")
                    obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                except Exception as e:
                    err_msg = str(e)
                    print(f"[Exploration] Exception on execute_pull: {err_msg}")
                    if 'predicate' in err_msg.lower():
                        predicate_missing = err_msg.strip()
                    exploration_done = True

            # 4. Further actions can be tested as needed (place, push, sweep, etc.)

        if predicate_missing:
            print(f"[Exploration] Missing predicate detected: {predicate_missing}")
        else:
            print("[Exploration] No explicit predicate error detected in basic drawer/handle interaction.")

        # =======================================================================
        #  (Optional) Now, you can continue to the task's actual oracle plan step-by-step
        #  using only primitive skills (as mapped in the provided plan).
        #  For demonstration, here's an example sequence:
        # =======================================================================

        # == Example plan: pick object from floor, open drawer, place object in drawer ==
        # 1. Move to object
        # 2. Pick object (from floor)
        # 3. Move to drawer (if not already there)
        # 4. Pick handle
        # 5. Pull (open drawer)
        # 6. Place object in drawer
        # 7. Push (close drawer)
        try:
            # 1. Move to object
            if current_robot_pos != object_location and object_location is not None:
                print(f"[Plan] execute_go: {current_robot_pos} → {object_location}")
                obs, reward, done = execute_go(env, task, current_robot_pos, object_location)
                current_robot_pos = object_location

            # 2. Pick the object
            if object_name is not None and current_robot_pos == object_location:
                print(f"[Plan] execute_pick: {object_name} at {object_location}")
                obs, reward, done = execute_pick(env, task, object_name, object_location)
            else:
                print(f"[Plan] Could not pick object: {object_name} @ {object_location}")

            # 3. Move to drawer location
            if current_robot_pos != drawer_location:
                print(f"[Plan] execute_go: {current_robot_pos} → {drawer_location}")
                obs, reward, done = execute_go(env, task, current_robot_pos, drawer_location)
                current_robot_pos = drawer_location

            # 4. Pick the handle
            print(f"[Plan] execute_pick: {handle_name} at {drawer_location}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)

            # 5. Pull (open the drawer)
            print(f"[Plan] execute_pull: Drawer={drawer_name} Handle={handle_name} at {drawer_location}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)

            # 6. Place the object into the drawer
            print(f"[Plan] execute_place: {object_name}, Drawer={drawer_name}, at {drawer_location}")
            obs, reward, done = execute_place(env, task, object_name, drawer_name, drawer_location)

            # 7. Push to close the drawer
            print(f"[Plan] execute_push: Drawer={drawer_name} at {drawer_location}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)

            print("[Plan] Finished demonstration plan.")

        except Exception as plan_e:
            print(f"[Plan] Exception during plan execution: {plan_e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
