# run_skeleton_task.py (Executable exploration-aware version)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, 
       extended to explore the environment for missing predicates or facts.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # The exploration phase is intended to probe for missing predicates or needed state facts,
        # in case the planner failed previously due to missing information, e.g., lock-known, temperature-known, durability.

        # We will perform available "exploration" skills to enumerate object properties that might be required for planning/actions.
        # Since we do NOT know the exact missing predicate, we try all plausible exploration phases.

        # Get list of all location/object names (best-effort since observation is only partially provided).
        # Positions likely contains object and location keys. For this demo, assume keys like 'drawer', 'drawer_handle', 'object_1', 'table', etc.
        # You'd usually parse (:objects ...) and (:init ...) for full knowledge, but we use what get_object_positions() returns.

        # Heuristics to partition objects vs. locations vs. handles:
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []

        for name in positions:
            lname = name.lower()
            if 'drawer' in lname and 'handle' not in lname:
                drawer_names.append(name)
            elif 'handle' in lname:
                handle_names.append(name)
            elif 'table' in lname or 'room' in lname or 'loc' in lname or 'floor' in lname:
                location_names.append(name)
            elif 'object' in lname or 'cube' in lname or 'ball' in lname:
                object_names.append(name)

        # Fallbacks for empty lists (best-effort exploration)
        if not location_names:
            # Guess from positions: treat all non-object, non-handle as possible locations
            for name in positions:
                if name not in object_names and name not in handle_names and name not in drawer_names:
                    location_names.append(name)

        # Also add drawers as locations if not already
        for dname in drawer_names:
            if dname not in location_names:
                location_names.append(dname)

        # Try to find the robot's starting location.
        # Since observation is missing, we may try to guess; otherwise, just pick the first location.
        if location_names:
            robot_init_loc = location_names[0]
        else:
            robot_init_loc = None

        # === EXPLORATION PHASE ===
        # We iterate through all provided skills for potential knowledge acquisition actions.
        # We use these to try and "reveal" missing predicates which are required, e.g., lock-known, durability, temperature, identified, etc.
        # We invoke the actions carefully, catching exceptions in case parameters do not match.

        print("[Exploration] Starting exploration actions to uncover missing predicates...")

        # 1. Try to move (go) to each location, identifying/detecting all objects in that location:
        for loc in location_names:
            if loc == robot_init_loc:
                continue
            try:
                print(f"[Exploration] Moving robot from {robot_init_loc} to {loc} with execute_go.")
                obs, reward, done = execute_go(env, task, robot_init_loc, loc)
                robot_init_loc = loc
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

            # Optionally, sweep at this location
            for obj in object_names:
                try:
                    print(f"[Exploration] Sweeping {obj} at {loc} with execute_sweep.")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {e}")

        # 2. Try picking each object in each location to possibly acquire information like weight-known or durability-known
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting pick of {obj} at {loc} with execute_pick.")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try pulling on handles to test for lock-known predicates.
        for dname in drawer_names:
            for hname in handle_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting pull on {dname} handle {hname} at {loc} with execute_pull.")
                        obs, reward, done = execute_pull(env, task, dname, hname, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try pushing each drawer at each location to test drawer close.
        for dname in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting push of drawer {dname} at {loc} with execute_push.")
                    obs, reward, done = execute_push(env, task, dname, loc)
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

        # 5. Try placing each object into each drawer at each location.
        for obj in object_names:
            for dname in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting place {obj} in {dname} at {loc} with execute_place.")
                        obs, reward, done = execute_place(env, task, obj, dname, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {e}")

        # 6. Try execute_gripper primitive if supported
        try:
            print(f"[Exploration] Attempting execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for predicate discovery.")

        # After this exploration, the simulation log (or subsequent planning/validation) should reveal what was missing
        # and thereby provide information on the missing predicates for further plan refinement.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
