# run_skeleton_task.py (Filled in as per task requirements)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined primitive skills only

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions = {'obj_1': (x,y,z), ...}
        positions = get_object_positions()

        # Exploration phase: dynamically check which predicates are missing in the domain
        # The goal: Call the exploration-related available primitive skills based on the available skill names

        # List of available primitives (according to provided "Available Skill Names")
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # We need to perform an "exploration" step to detect missing predicates

        # For demonstration, we'll run all possible exploration primitives on all available objects/locations
        print("[Exploration] Beginning exploration actions to identify possible missing predicates or preconditions.")

        # Gather likely lists of objects, drawers, locations based on positions or dummy names
        object_names = [name for name in positions if 'drawer' not in name and 'handle' not in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = list({tuple(pos) for pos in positions.values()})
        # Fallback: Use object name suffixes as locations if available
        if not location_names:
            # fallback: ['table', 'floor', ...] (guess based on names)
            location_names = ['table', 'floor']

        # Step 1: Try moving to each location to see if any predicate becomes known/identified
        cur_location = None
        try:
            # Attempt to deduce initial robot location from observation (if available)
            for name in positions:
                if 'robot' in name:
                    cur_location = positions[name]
                    break
        except Exception:
            cur_location = None

        # For the demo, just pick a random start location if missing
        if cur_location is None and location_names:
            cur_location = location_names[0]

        # Now, run each exploration skill if available, as inspired by exploration domain's actions

        # 1. execute_go: Move robot between locations (simulate exploration)
        # 2. execute_pick: Try to pick an object to see if predicates like "identified", "weight-known", etc. are needed
        # 3. execute_pull: Pull handles on drawers to check lock status
        # 4. execute_sweep, execute_gripper, execute_push, execute_place can be similarly called if required

        # We'll proceed with these in sequence

        # Exploration: Move robot around to all known locations
        print("[Exploration] Moving robot to each location to test 'robot-at' and to help identify missing predicates.")
        prev_location = cur_location

        for tgt_location in location_names:
            # Call execute_go (expects (from, to))
            if prev_location != tgt_location:
                try:
                    print(f"[Exploration] execute_go from {prev_location} to {tgt_location}")
                    obs, reward, done, info = execute_go(env, task, prev_location, tgt_location)
                    prev_location = tgt_location
                except Exception as e:
                    print(f"[Exploration] Warning: execute_go failed ({prev_location} -> {tgt_location}): {e}")

        # Exploration: For each object, try execute_pick at each location
        for obj in object_names:
            for location in location_names:
                try:
                    print(f"[Exploration] Attempting execute_pick on {obj} at location {location}")
                    obs, reward, done, info = execute_pick(env, task, obj, location)
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {location}: {e}")

        # Exploration: For each drawer, try execute_pull with possible handles
        for drawer in drawer_names:
            # Try to find a handle object for the drawer, if present in positions
            handle = None
            for obj in positions:
                if 'handle' in obj and drawer in obj:
                    handle = obj
                    break
            if not handle:
                # fallback: just try any 'handle' object if present
                handles = [obj for obj in positions if 'handle' in obj]
                handle = handles[0] if handles else None
            for location in location_names:
                try:
                    if handle:
                        print(f"[Exploration] Trying execute_pull on drawer {drawer} via handle {handle} at {location}")
                        obs, reward, done, info = execute_pull(env, task, drawer, handle, location)
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {drawer} at {location}: {e}")

        # (Optional) Try the other available skills as well for completeness:

        # Sweep - Try sweeping objects on each location
        for obj in object_names:
            for location in location_names:
                try:
                    print(f"[Exploration] Sweeping {obj} at {location}")
                    obs, reward, done, info = execute_sweep(env, task, obj, location)
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {location}: {e}")

        # Push - Try pushing all drawers open/closed at every location
        for drawer in drawer_names:
            for location in location_names:
                try:
                    print(f"[Exploration] Trying execute_push on {drawer} at {location}")
                    obs, reward, done, info = execute_push(env, task, drawer, location)
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {location}: {e}")

        # Place - Try placing all held objects in all drawers at every location
        for obj in object_names:
            for drawer in drawer_names:
                for location in location_names:
                    try:
                        print(f"[Exploration] Attempting execute_place of {obj} in {drawer} at {location}")
                        obs, reward, done, info = execute_place(env, task, obj, drawer, location)
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj} in {drawer} at {location}: {e}")

        # Gripper - Try gripper action
        try:
            print(f"[Exploration] Attempting execute_gripper")
            obs, reward, done, info = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check env, logs, or task observations for new/missing predicates indicated by failed steps or action preconditions.")

        # After this exploration phase, use feedback or task observations to update the problem/domain with required missing predicates.

        # --- End of dynamic exploration phase ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
