# run_skeleton_task.py (Filled Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # =============================== #
        # =========== PLAN ============== #
        # =============================== #
        # This plan is designed to (a) explore for missing lock knowledge (predicate)
        # and (b) manipulate handle1 as per feedback.
        # All skill invocations use only functions provided by skill_code.

        # For demonstration, typical objects, drawers, handles and locations
        # These names should match actual environment names, adjust if needed.
        drawer = 'drawer1'
        handle = 'handle1'
        object_name = handle    # For lock-known exploration we assume exploring the handle (drawer handle)
        location = 'location_drawer1'  # Robot's navigation goal location for the drawer
        initial_location = 'robot_start'  # Initial location of the robot (update as appropriate)

        # Try to get real names for location/object/etc if available
        # Use whatever actual names appear in positions
        try:
            # Try to infer drawer, handle, locations using provided positions keys
            for key in positions:
                if 'drawer' in key:
                    drawer = key
                if 'handle' in key:
                    handle = key
                if 'location' in key and 'drawer' in key:
                    location = key
                if 'robot' in key and 'location' in key:
                    initial_location = positions[key]
        except Exception as e:
            print("[Exploration] Could not infer names from positions:", e)

        # Fallback values for demonstration if real keys missing
        if drawer not in positions:
            drawer = list(positions.keys())[0]
        if handle not in positions:
            handle = list(positions.keys())[0]
        if location not in positions:
            location = list(positions.keys())[0]

        # State logging
        print(f"[Task] Using drawer: {drawer}")
        print(f"[Task] Using handle: {handle}")
        print(f"[Task] Using target/location: {location}")

        # Phase 1 - Exploration for Missing Predicate (lock-known) via execute_pull
        print("[Exploration] Trying to acquire lock-known predicate (exploring lock state)...")

        # Step 1.1: If robot not at drawer location, move there
        try:
            robot_pos = None
            for k in positions:
                if 'robot' in k and 'location' in k:
                    robot_pos = positions[k]
                    break
            # If the robot is not at location, call execute_go
            if robot_pos != positions[location]:
                print("[Task] Moving robot to drawer location.")
                obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=positions[location])
        except Exception as e:
            print("[Task] Exception while moving to drawer location:", e)

        # Step 1.2: Pick up the handle before execute_pull
        print("[Task] Picking up handle object for drawer.")
        try:
            obs, reward, done = execute_pick(env, task, object=handle, location=positions[location])
            if done:
                print("[Task] Picked up handle. (holding handle1)")
        except Exception as e:
            print("[Task] Failed to pick up handle:", e)

        # Step 1.3: Attempt to execute_pull (explore lock-known)
        print("[Exploration] Executing pull action to attempt lock-known discovery (should trigger missing predicate).")
        try:
            obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=positions[location])
            if done:
                print("[Exploration] Pull succeeded. lock-known predicate may now be true.")
        except Exception as e:
            print("[Exploration] Pull/exploration failed (possible missing lock-known precondition):", e)

        # Phase 2: Continue with demonstration or further plan as needed
        # (example: open drawer, place an object, etc)
        # Here, we demonstrate continued use of skills after lock-known exploration

        # Example - Place an object in the drawer (if the drawer is open)
        # Only execute if previous steps are done, and handle is available
        print("[Task] Trying to place handle into drawer (if open).")
        try:
            obs, reward, done = execute_place(env, task, object=handle, drawer=drawer, location=positions[location])
            if done:
                print("[Task] Placed handle into drawer.")
        except Exception as e:
            print("[Task] Failed to place handle into drawer:", e)

        # Note: The above control flow assumes the predicate (holding handle1) feedback,
        # that missing lock-known must be explored via execute_pull on the handle (drawer handle).
        # If further task steps are needed, append action calls as per oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
