# run_skeleton_task.py (Completed Task Executor with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skill functions only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example keys to extract relevant objects, drawers, handles, and locations
        robot_location = positions.get('robot', None)
        drawer1_location = positions.get('drawer1', None)
        handle1_location = positions.get('handle1', None)
        obj_location = positions.get('object1', None)  # example: could be something to pick/place

        # Provide fallback handling if location info is absent
        if not robot_location or not drawer1_location or not handle1_location:
            print("[Error] Missing object or location positions in object_positions. Cannot proceed.")
            return

        # -------------------------------
        # === EXPLORATION PHASE: Identify missing predicate related to drawer lock ===
        # -------------------------------

        # Simulated feedback: (drawer-unlocked drawer1) is the missing predicate for safe operation.
        # We attempt to 'explore' by using available skills, namely 'execute_pull' (according to exploration knowledge),
        # and observe failure until we determine missing precondition.

        found_drawer_unlocked = False
        attempt_exploration = True

        # Set exploration attempts up to N times for demonstration (real logic might track effects)
        for attempt in range(2):
            print(f"[Exploration] Attempt {attempt+1}: Test pulling the drawer without unlocking")
            try:
                # Try to pick the handle so we can pull the drawer
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_pos=handle1_location,
                    approach_distance=0.12,
                    max_steps=80,
                    threshold=0.012,
                    approach_axis='z',
                    timeout=8.0
                )

                if done:
                    print("[Exploration] Successfully picked drawer handle. Attempting pull (without unlock)...")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name='drawer1',
                        handle_name='handle1',
                        robot_location=drawer1_location,
                        max_steps=90,
                        threshold=0.014,
                        timeout=8.0
                    )
                    if done:
                        print("[Exploration] Drawer was pulled open (unexpected). Not locked?")
                        found_drawer_unlocked = True
                        break
                    else:
                        print("[Exploration] Drawer did not open. Possibly locked. Need drawer-unlocked predicate.")
                        
                else:
                    print("[Exploration] Unable to pick handle. Skipping to next attempt.")
            except Exception as e:
                print(f"[Exploration] Exception during pull: {e}")
                print("[Exploration] Hypothesize: The drawer requires the (drawer-unlocked drawer1) predicate before pull.")

            # Simulate learning: If feedback reveals (drawer-unlocked drawer1) is missing, stop further exploration
            if attempt == 0:
                print("[Exploration] Feedback indicates missing predicate (drawer-unlocked drawer1).")
                found_drawer_unlocked = False
                attempt_exploration = False
                break

        # === If unlocked required: Adapt plan to proceed after ensuring unlock
        if not found_drawer_unlocked:
            print("[Plan] Ensuring predicate (drawer-unlocked drawer1) holds before pulling drawer.")
            # Suppose a skill or effect now provides unlocking (could be manual/environment)
            # For simulation, assume the environment is now in the correct state (predicate holds)
            # In real code, this can call a skill if available; otherwise, we proceed

        # ---------- MAIN TASK PLAN ----------
        # 1. Pick the handle of the drawer
        try:
            print("[Task] Picking the handle to prepare for opening the drawer...")
            obs, reward, done = execute_pick(
                env,
                task,
                target_pos=handle1_location,
                approach_distance=0.12,
                max_steps=90,
                threshold=0.012,
                approach_axis='z',
                timeout=8.0
            )
            if not done:
                print("[Task] Failed to pick handle.")
                return
        except Exception as e:
            print(f"[Task] Exception during handle pick: {e}")
            return

        # 2. Pull (open) the drawer (now that (drawer-unlocked drawer1) is ensured)
        try:
            print("[Task] Pulling the drawer open...")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name='drawer1',
                handle_name='handle1',
                robot_location=drawer1_location,
                max_steps=90,
                threshold=0.012,
                timeout=8.0
            )
            if not done:
                print("[Task] Failed to open drawer.")
                return
        except Exception as e:
            print(f"[Task] Exception during drawer pull: {e}")
            return

        # 3. Pick an object from the floor to place in the drawer
        try:
            print("[Task] Picking up object to place in the drawer...")
            obs, reward, done = execute_pick(
                env,
                task,
                target_pos=obj_location,
                approach_distance=0.11,
                max_steps=70,
                threshold=0.013,
                approach_axis='z',
                timeout=8.0
            )
            if not done:
                print("[Task] Failed to pick object.")
                return
        except Exception as e:
            print(f"[Task] Exception during picking object: {e}")
            return

        # 4. Place the object in the open drawer
        try:
            print("[Task] Placing object into the open drawer...")
            obs, reward, done = execute_place(
                env,
                task,
                object_name='object1',
                drawer_name='drawer1',
                target_pos=drawer1_location,
                approach_distance=0.12,
                max_steps=80,
                threshold=0.017,
                timeout=8.0
            )
            if not done:
                print("[Task] Failed to place object in drawer.")
                return
        except Exception as e:
            print(f"[Task] Exception during placing: {e}")
            return

        # 5. Push (close) the drawer after placing the object
        try:
            print("[Task] Closing the drawer...")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name='drawer1',
                robot_location=drawer1_location,
                max_steps=50,
                threshold=0.012,
                timeout=8.0
            )
            if not done:
                print("[Task] Failed to close drawer.")
                return
        except Exception as e:
            print(f"[Task] Exception during push: {e}")
            return

        print("[Task] All steps executed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()