# run_skeleton_task.py (Filled according to instructions)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, upgraded for lock-known exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # EXPLORATION PHASE - Find missing predicate (using skill primitives only)
        # The feedback was: (drawer-unlocked drawer1) is missing. We don't know if drawer1 is locked/unlocked.
        # We must execute an exploration skill to discover lock-known status for the drawer/handle.
        #
        # Given exploration domain and available skills, the action representing "pull and find lock status"
        # is "execute_pull" (assuming it triggers lock-known by effect/failure).

        # 1. Identify the drawer and its handle from the positions
        # (Assume there is only one drawer and handle in this setting)
        drawer_name = None
        handle_name = None
        location_name = None
        for obj in positions.keys():
            if 'drawer' in obj and 'handle' not in obj:
                drawer_name = obj
            elif 'handle' in obj:
                handle_name = obj

        # Try to deduce the robot's current location
        # (Assume positions dictionary will have a value for robot position if available)
        if 'robot' in positions:
            location_name = positions['robot']['location'] if isinstance(positions['robot'], dict) and 'location' in positions['robot'] else None
        else:
            # Fallback: search for something that looks like a location entry
            location_name = None

        # If no explicit location, try a known location string (fallback)
        if location_name is None:
            for k in positions.keys():
                if 'room' in k or 'location' in k or 'loc' in k:
                    location_name = k
                    break
            # Ultimate fallback
            if location_name is None:
                location_name = 'location1'

        # Print detected items for debug
        print(f"Detected drawer: {drawer_name}, handle: {handle_name}, location: {location_name}")

        # Ensure variables are all there
        if drawer_name is None or handle_name is None:
            print('[Exploration] Could not find drawer or handle in object positions. Aborting exploration.')
            return

        # 2. Go to the drawer's location (if needed) -- use execute_go
        # Assume the drawer or handle has a position that can map to a location
        robot_current_location = location_name
        target_location = None
        for k in positions.keys():
            if k == drawer_name and 'location' in positions[k]:
                target_location = positions[k]['location']
                break
            elif k == handle_name and 'location' in positions[k]:
                target_location = positions[k]['location']
                break
        if not target_location:
            # Fallback to using generic
            target_location = location_name

        # Move to the handle/drawer location
        print(f"[Exploration] Moving robot to drawer/handle location: {target_location}")
        try:
            obs, reward, done, info = execute_go(env, robot_current_location, target_location)
        except Exception as e:
            print(f"[Exploration] Error when calling execute_go: {e}")

        # 3. Pick up the handle with execute_pick
        print("[Exploration] Picking up the handle to interact with drawer...")
        try:
            obs, reward, done, info = execute_pick(env, handle_name, target_location)
        except Exception as e:
            print(f"[Exploration] Error when calling execute_pick: {e}")

        # 4. Try to pull the drawer with execute_pull (this serves as lock-known exploration)
        print("[Exploration] Trying to pull the drawer to learn lock status...")
        try:
            obs, reward, done, info = execute_pull(env, drawer_name, handle_name, target_location)
            print("[Exploration] execute_pull executed successfully. Drawer lock status can now be inferred.")
        except Exception as e:
            print(f"[Exploration] Error when calling execute_pull: {e}")
            print("[Exploration] The attempt to pull may have failed due to the drawer being locked, lock-known is now established.")

        # Exploration Phase End (lock-known predicate inferred for drawer/handle)

        # === Now, proceed to normal task plan (oracle) ===
        # The goal: (drawer-unlocked drawer1), i.e., unlocked state must be established.
        # If unlocked, proceed to open, place, etc.

        # The following is generic/sketch code - replace placeholders with actual environment variables as appropriate.
        # For demonstration, we attempt to finish "open drawer" sequence:
        
        # 5. If the drawer is unlocked and still closed, open it.
        print("[Task] Attempting to open the unlocked drawer.")
        try:
            # (Assume after exploration, we are still holding the handle)
            obs, reward, done, info = execute_pull(env, drawer_name, handle_name, target_location)
            print("[Task] Drawer open attempted.")
        except Exception as e:
            print(f"[Task] Error trying to open drawer after unlock: {e}")

        # 6. Place an object into the drawer (if provided), e.g., ball1
        place_obj = None
        for obj in positions.keys():
            if 'ball' in obj or 'object' in obj:
                place_obj = obj
                break
        if place_obj is not None:
            print(f"[Task] Placing object {place_obj} into drawer {drawer_name}.")
            try:
                obs, reward, done, info = execute_place(env, place_obj, drawer_name, target_location)
                print("[Task] Object placed in drawer.")
            except Exception as e:
                print(f"[Task] Error placing object in drawer: {e}")
        else:
            print("[Task] No object found to place into drawer.")

        # 7. Push drawer closed (optional, for task completion)
        print("[Task] Attempting to close the drawer.")
        try:
            obs, reward, done, info = execute_push(env, drawer_name, target_location)
            print("[Task] Drawer closed.")
        except Exception as e:
            print(f"[Task] Error closing drawer: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()