# run_skeleton_task.py (Completed Version with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All primitive skills are used as imported

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Identify the objects, drawers, handles, and locations from the environment ---
        # These variable names are consistent with the PDDL domain and common conventions
        try:
            # You might need to adapt the keys if your simulation uses different naming
            drawer1 = 'drawer1'
            handle1 = 'handle1'
            location_a = 'location_a'
            location_b = 'location_b'
            # Assume object1 to place, which is on the floor:
            object1 = 'object1'
            robot_home = 'home'
        except Exception as e:
            print("[Error] Problem determining object names from positions:", e)
            return

        # ASSUME: the plan involves:
        # 1. Robot identifies drawer state (is it closed? locked? etc).
        # 2. If (drawer-closed drawer1) predicate is missing, perform exploration to check.
        # 3. Then, execute the relevant plan (e.g. open drawer, pick, place) using primitive skills.

        # ------------------------- Exploration Phase -----------------------------
        # The missing predicate is (drawer-closed drawer1). To confirm its truth value,
        # we can attempt to use available skills that would allow us to discover its state.
        # Since execute_pull requires drawer to be closed and unlocked,
        # and (drawer-closed drawer1) may be missing, we attempt to explore.

        print("[Exploration] Checking if (drawer-closed drawer1) is present or discoverable.")

        # Here, we'd do something like "execute_pull" to check lock-known, or similar;
        # But since only predefined skills can be called (no new skill), we use available ones.

        try:
            # Move robot to drawer1's location if not there
            robot_current_location = robot_home  # start at home
            drawer1_location = location_a  # assume drawer1 at location_a
            if robot_current_location != drawer1_location:
                print("[Exploration] Moving robot from", robot_current_location, "to", drawer1_location)
                obs, reward, done = execute_go(env, task, robot_current_location, drawer1_location)
                robot_current_location = drawer1_location

            # Try to pull the drawer to see if it is closed (if not open yet)
            # The exploration PDDL shows "execute_pull" as a way to update lock-known,
            # but in our execution, we focus on effects that help us check closed state.
            # We try to grab the handle then pull.

            # First, pick the handle
            print("[Exploration] Picking handle:", handle1, "at", drawer1_location)
            obs, reward, done = execute_pick(env, task, handle1, drawer1_location)
            if done:
                print("[Exploration] Picked handle - now attempting to pull to check (drawer-closed drawer1)")
                # Try to pull the drawer 
                obs, reward, done = execute_pull(env, task, drawer1, handle1, drawer1_location)
                if done:
                    print("[Exploration] Pulled handle - drawer should now be open, (drawer-closed drawer1) is FALSE after this.")
                else:
                    print("[Exploration] Could not pull drawer (perhaps already open/unlocked).")
            else:
                print("[Exploration] Could not pick handle (maybe already holding or missing).")
        except Exception as e:
            print("[Exploration] Exception during drawer exploration:", e)

        # At this point, we explored the drawer state,
        # and in an oracle plan we would use this knowledge to ensure correct predicates.

        # ---------------------- Begin Main Plan Execution -----------------------

        try:
            # 1. If robot is not at drawer location, move to drawer location
            robot_current_location = drawer1_location  # already moved in exploration phase

            # 2. If drawer is not open, pick handle and open drawer
            # This is handled in exploration above; we can check obs for state if needed.

            # 3. Pick up the object from the floor (object1 at same location)
            print("[Task] Picking up object:", object1, "at", drawer1_location)
            obs, reward, done = execute_pick(env, task, object1, drawer1_location)
            if done:
                print("[Task] Picked up object.")

                # 4. Place object into the drawer
                print("[Task] Placing object:", object1, "into drawer:", drawer1, "at", drawer1_location)
                obs, reward, done = execute_place(env, task, object1, drawer1, drawer1_location)
                if done:
                    print("[Task] Placed object in drawer.")

                    # 5. Push drawer closed using the primitive
                    print("[Task] Closing drawer:", drawer1)
                    obs, reward, done = execute_push(env, task, drawer1, drawer1_location)
                    if done:
                        print("[Task] Drawer closed.")
                    else:
                        print("[Task] Could not close drawer for unknown reason.")
                else:
                    print("[Task] Could not place object into drawer.")
            else:
                print("[Task] Could not pick up object from floor.")

        except Exception as e:
            print("[Task] Exception during main task execution:", e)

        # -------------------------- End of Plan -------------------------------

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()