# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine primitives. Use skills as provided.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dictionary of {object_name: position_tuple}
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Warning] Unable to retrieve object positions:", str(e))
            positions = {}

        # ===================== EXPLORATION PHASE =======================
        # The robot is to explore to discover the missing predicate ("robot-at dice1").
        print("[Exploration] Begin exploration phase to find missing predicates...")

        # Example logic: try to move the robot to every location in the environment
        # to see which predicates (e.g. robot-at <loc>) are set/discovered.

        # We'll try to use only the available skills, especially navigation skills.
        # Suppose our environment has locations named as in positions.keys();

        # Known available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        #   'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # From exploration PDDL, robot-at is per-location: (robot-at ?loc)
        # In the feedback, (robot-at dice1) is mentioned.

        # Let's gather all locations from either positions or descriptions/obs (if available).
        location_names = []
        # Try to find plausible location names from positions, descriptions, or obs
        if positions:
            # Assume all object keys ending with digits like "dice1" may represent locations, too
            for name in positions:
                if "loc" in name or "dice" in name or "room" in name or "drawer" in name:
                    location_names.append(name)
        if not location_names:
            # As fallback, try from description strings if structure known
            if 'locations' in descriptions:
                location_names = descriptions['locations']
            elif 'location_names' in descriptions:
                location_names = descriptions['location_names']
            else:
                # Fallback: Use some defaults
                location_names = ["dice1", "dice2", "table", "drawer1"]

        # Remove duplicates and obviously non-location entries
        location_names = list(set(location_names))

        print("[Exploration] Detected possible locations:", location_names)

        # Now, systematically try to move the robot to all these locations using execute_go
        # (assuming the robot starts at the first location; we will always track its current location)
        exploration_done = False
        current_location = None

        # Get initial robot location, if available in obs or state
        # Try to deduce initial robot location (e.g., from obs keys or via skill)
        for key in ('robot_location', 'robot-at', 'robot_pos', 'robot_loc'):
            if key in obs:
                current_location = obs[key]
                break
        if not current_location and location_names:
            # Fallback: start at the first location
            current_location = location_names[0]

        print("[Exploration] Robot starting at location:", current_location)

        discovered_robot_at = []

        for loc in location_names:
            if loc == current_location:
                print(f"[Exploration] Robot already at {loc}")
                discovered_robot_at.append(loc)
                continue
            try:
                print(f"[Exploration] Attempting to move robot: {current_location} -> {loc} using execute_go...")
                obs, reward, done = execute_go(
                    env, task,
                    from_location=current_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                print(f"[Exploration] Action returned. Robot should now be at {loc}.")
                current_location = loc
                discovered_robot_at.append(loc)
            except Exception as e:
                print(f"[Exploration] Could not go to {loc}: {e}")

        print("[Exploration] Explored locations (robot-at predicates discovered):", discovered_robot_at)
        print("[Exploration] Looking for missing predicates related to robot-at...")
        # If 'dice1' is in locations and not marked as 'robot-at', it points to a missing predicate.

        # For demonstration, we log if 'robot-at dice1' can now be verified.
        if "dice1" in discovered_robot_at:
            print("[Exploration] Missing predicate (robot-at dice1) has now been explored/discovered.")
        else:
            print("[Exploration] WARNING: (robot-at dice1) could not be confirmed, please check environment locations.")


        # ===================== TASK COMPLETION PHASE =======================
        # (Placeholder) This is where you'd execute the actual oracle plan steps
        # by calling the appropriate primitives (skills). Example:
        #
        # obs, reward, done = execute_pick(env, task, object_name, position)
        # obs, reward, done = execute_place(env, task, object_name, drawer_name, position)
        #
        # You could loop through an oracle plan as a sequence of skill calls here.
        #
        # For now, we finish after the exploration phase for predicate discovery.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
