# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be explored.
        # We will use available skills to explore the state of drawer1.

        # Assumptions for object names and locations
        # You may need to adjust these names based on your environment
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = 'loc_robot'
        drawer_location = 'loc_drawer'
        handle_location = 'loc_handle'
        # The actual keys in positions may differ; adjust as needed

        # Step 1: Move robot to drawer location if not already there
        try:
            robot_pos = positions.get('robot', None)
            drawer_pos = positions.get(drawer_name, None)
            if robot_pos is not None and drawer_pos is not None:
                if robot_pos != drawer_pos:
                    print(f"[Exploration] Moving robot from {robot_pos} to {drawer_pos}")
                    obs, reward, done = execute_go(env, task, robot_pos, drawer_pos)
                    if done:
                        print("[Exploration] Task ended during move!")
                        return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")

        # Step 2: Try to pick the handle (if required for pull)
        try:
            print(f"[Exploration] Attempting to pick handle: {handle_name}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
            if done:
                print("[Exploration] Task ended during pick!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # Step 3: Try to pull the drawer to check lock/closed state
        try:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} with handle: {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")

        # Step 4: If pull fails, the drawer may be locked or closed; try to push (close) and then pull again
        try:
            print(f"[Exploration] Attempting to push (close) drawer: {drawer_name}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            if done:
                print("[Exploration] Task ended during push!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during push: {e}")

        # Step 5: Try to pull again after push
        try:
            print(f"[Exploration] Attempting to pull drawer again: {drawer_name} with handle: {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            if done:
                print("[Exploration] Task ended during second pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during second pull: {e}")

        # Step 6: Exploration complete, check for missing predicate
        print("[Exploration] Exploration phase complete. If drawer could not be pulled, predicate (drawer-closed drawer1) may be missing or not set.")

        # === Main Task Plan (Example) ===
        # Insert your oracle plan here, using only the available skills.
        # For example, to pick an object and place it in the drawer:
        # (You may need to adjust object and location names as per your environment.)

        # Example: Pick up an object from the floor and place it in the drawer
        object_to_pick = 'object1'
        object_location = 'loc_object1'
        try:
            print(f"[Task] Moving to object location: {object_location}")
            obs, reward, done = execute_go(env, task, drawer_location, object_location)
            if done:
                print("[Task] Task ended during move to object!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to object: {e}")

        try:
            print(f"[Task] Picking up object: {object_to_pick}")
            obs, reward, done = execute_pick(env, task, object_to_pick, object_location)
            if done:
                print("[Task] Task ended during pick object!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick object: {e}")

        try:
            print(f"[Task] Moving to drawer location: {drawer_location}")
            obs, reward, done = execute_go(env, task, object_location, drawer_location)
            if done:
                print("[Task] Task ended during move to drawer!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to drawer: {e}")

        try:
            print(f"[Task] Placing object: {object_to_pick} into drawer: {drawer_name}")
            obs, reward, done = execute_place(env, task, object_to_pick, drawer_name, drawer_location)
            if done:
                print("[Task] Task ended during place!")
                return
        except Exception as e:
            print(f"[Task] Exception during place: {e}")

        print("[Task] Oracle plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()