# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Find the missing predicate (robot-at drawer-area) ---
        # The feedback indicates that (robot-at drawer-area) is missing or not being set.
        # We will use the available skills to move the robot to 'drawer-area' and check the state.

        # Assumptions:
        # - 'drawer-area' is a location in the environment.
        # - The robot's current location and the list of locations are available.
        # - The skill 'execute_go' is used to move the robot between locations.

        # Find the robot's current location and the 'drawer-area' location
        # This may require parsing the positions dictionary or using task/obs info.
        # For demonstration, we assume positions contains keys for locations.

        # Try to find the robot's current location
        robot_location = None
        drawer_area_location = None

        # Try to infer location names from positions
        for key in positions:
            if 'robot' in key and positions[key].get('is_robot', False):
                robot_location = positions[key].get('location', None)
            if key == 'drawer-area' or 'drawer_area' in key:
                drawer_area_location = key

        # Fallback: If not found, use default names
        if robot_location is None:
            # Try to get from obs or descriptions
            robot_location = 'start_area'
        if drawer_area_location is None:
            drawer_area_location = 'drawer-area'

        print(f"[Exploration] Robot current location: {robot_location}")
        print(f"[Exploration] Target location (drawer-area): {drawer_area_location}")

        # Use execute_go to move the robot to drawer-area
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {drawer_area_location} using execute_go...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {drawer_area_location}.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # After moving, check if the predicate (robot-at drawer-area) is now true
        # This may require checking the environment state or observation
        # For demonstration, print the observation or relevant state
        try:
            current_obs = task.get_observation()
            print(f"[Exploration] Current observation after move: {current_obs}")
            # Optionally, check for 'robot-at drawer-area' in the state if possible
            if hasattr(task, 'get_state_predicates'):
                predicates = task.get_state_predicates()
                if ('robot-at', drawer_area_location) in predicates:
                    print("[Exploration] Predicate (robot-at drawer-area) is now true.")
                else:
                    print("[Exploration] Predicate (robot-at drawer-area) is still missing.")
            else:
                print("[Exploration] Cannot directly check predicates; please verify in the environment.")
        except Exception as e:
            print(f"[Exploration] Error checking observation/state: {e}")

        # --- End of Exploration Phase ---

        # TODO: Continue with further task steps or exploration as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()