# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Find missing predicate (robot-at drawer-area) ---
        # Feedback indicates that (robot-at drawer-area) is missing or needs to be explored.
        # We will use the available skills to move the robot to 'drawer-area' and check the state.

        # Try to find a location named 'drawer-area' in the positions dictionary
        drawer_area_key = None
        for key in positions:
            if 'drawer' in key and 'area' in key:
                drawer_area_key = key
                break
        if drawer_area_key is None:
            # Fallback: try common names
            for key in positions:
                if key == 'drawer-area':
                    drawer_area_key = key
                    break
        if drawer_area_key is None:
            print("[Exploration] Could not find 'drawer-area' in object positions. Exploration aborted.")
            return

        drawer_area_pos = positions[drawer_area_key]

        # Find current robot location
        robot_location_key = None
        for key in positions:
            if 'robot' in key and 'area' in key:
                robot_location_key = key
                break
        if robot_location_key is None:
            # Fallback: try common names
            for key in positions:
                if key == 'robot-area':
                    robot_location_key = key
                    break
        if robot_location_key is None:
            # If not found, just pick any location (assume only one robot location)
            for key in positions:
                if 'area' in key:
                    robot_location_key = key
                    break
        if robot_location_key is None:
            print("[Exploration] Could not determine robot's current area. Exploration aborted.")
            return

        robot_area_pos = positions[robot_location_key]

        print(f"[Exploration] Robot current area: {robot_location_key} at {robot_area_pos}")
        print(f"[Exploration] Target drawer area: {drawer_area_key} at {drawer_area_pos}")

        # Use the available skill 'execute_go' to move the robot from its current area to the drawer area
        # The skill signature is likely: execute_go(env, from_location, to_location)
        try:
            print(f"[Exploration] Executing 'execute_go' from {robot_location_key} to {drawer_area_key}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location_key,
                to_location=drawer_area_key
            )
            print(f"[Exploration] Robot moved to {drawer_area_key}.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # After moving, check if the predicate (robot-at drawer-area) is now true in the environment
        # This may require querying the environment or checking the observation
        # For demonstration, we print the observation and look for the predicate
        try:
            current_obs = task.get_observation()
            print("[Exploration] Current observation after moving to drawer-area:")
            print(current_obs)
            # If the observation is a dict or has a 'predicates' field, check for 'robot-at drawer-area'
            if isinstance(current_obs, dict):
                predicates = current_obs.get('predicates', [])
                if any('robot-at' in p and drawer_area_key in p for p in predicates):
                    print(f"[Exploration] Predicate (robot-at {drawer_area_key}) is now true.")
                else:
                    print(f"[Exploration] Predicate (robot-at {drawer_area_key}) is still missing.")
            else:
                # Fallback: just print the observation
                print(current_obs)
        except Exception as e:
            print(f"[Exploration] Error checking observation: {e}")

        # --- End of Exploration Phase ---

        # TODO: Continue with the main oracle plan using the available skills as needed.
        # For this exploration task, we stop here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()