# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The missing predicate is likely related to the robot's location at 'drawer-area'.
        # We will use the available skills to explore and confirm the robot can be at 'drawer-area'.

        # Try to move the robot to 'drawer-area' using execute_go
        # Assume positions contains keys for locations, e.g., 'start', 'drawer-area'
        # If not, fallback to any available location keys

        # Find current robot location and drawer-area
        robot_location = None
        drawer_area = None
        for loc in positions:
            if 'robot' in loc or 'start' in loc:
                robot_location = loc
            if 'drawer' in loc and 'area' in loc:
                drawer_area = loc
        if robot_location is None:
            # Fallback: pick any location as start
            robot_location = list(positions.keys())[0]
        if drawer_area is None:
            # Fallback: pick any location with 'drawer' in name
            for loc in positions:
                if 'drawer' in loc:
                    drawer_area = loc
                    break
        if drawer_area is None:
            # Fallback: pick any location
            drawer_area = list(positions.keys())[-1]

        print(f"[Exploration] Moving robot from {robot_location} to {drawer_area} to check (robot-at drawer-area) predicate.")

        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {drawer_area}. Predicate (robot-at {drawer_area}) is now true.")
        except Exception as e:
            print(f"[Exploration] Failed to move robot to {drawer_area}: {e}")

        # === Main Task Plan Execution ===
        # Here, you would execute the oracle plan step-by-step using only the available skills.
        # For demonstration, let's assume the plan is:
        # 1. Move to drawer-area (already done above)
        # 2. Pull the drawer open (execute_pull)
        # 3. Pick an object from the floor (execute_pick)
        # 4. Place the object in the drawer (execute_place)
        # 5. Push the drawer closed (execute_push)

        # The following is a generic sequence using only available skills and positions.
        # You must adapt object and drawer names to your environment.

        # Find a handle object and a drawer
        handle_obj = None
        drawer_obj = None
        for obj in positions:
            if 'handle' in obj:
                handle_obj = obj
            if 'drawer' in obj:
                drawer_obj = obj
        if handle_obj is None:
            # Fallback: pick any object
            handle_obj = list(positions.keys())[0]
        if drawer_obj is None:
            drawer_obj = drawer_area  # fallback

        # 2. Pull the drawer open (requires holding the handle)
        print(f"[Task] Attempting to pick handle {handle_obj} at {drawer_area}")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_obj,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Picked handle {handle_obj}.")
        except Exception as e:
            print(f"[Task] Failed to pick handle {handle_obj}: {e}")

        print(f"[Task] Attempting to pull open drawer {drawer_obj} using handle {handle_obj} at {drawer_area}")
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_obj,
                handle_name=handle_obj,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Pulled open drawer {drawer_obj}.")
        except Exception as e:
            print(f"[Task] Failed to pull open drawer {drawer_obj}: {e}")

        # 3. Pick an object from the floor
        floor_obj = None
        for obj in positions:
            if 'object' in obj or 'item' in obj:
                floor_obj = obj
                break
        if floor_obj is None:
            # Fallback: pick any object not already used
            for obj in positions:
                if obj != handle_obj and obj != drawer_obj:
                    floor_obj = obj
                    break
        if floor_obj is None:
            floor_obj = handle_obj  # fallback

        print(f"[Task] Attempting to pick object {floor_obj} at {drawer_area}")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=floor_obj,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Picked object {floor_obj}.")
        except Exception as e:
            print(f"[Task] Failed to pick object {floor_obj}: {e}")

        # 4. Place the object in the drawer
        print(f"[Task] Attempting to place object {floor_obj} in drawer {drawer_obj} at {drawer_area}")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=floor_obj,
                drawer_name=drawer_obj,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Placed object {floor_obj} in drawer {drawer_obj}.")
        except Exception as e:
            print(f"[Task] Failed to place object {floor_obj} in drawer {drawer_obj}: {e}")

        # 5. Push the drawer closed
        print(f"[Task] Attempting to push drawer {drawer_obj} closed at {drawer_area}")
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_obj,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Pushed drawer {drawer_obj} closed.")
        except Exception as e:
            print(f"[Task] Failed to push drawer {drawer_obj} closed: {e}")

        print("[Task] Task execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()