# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from feedback and example) ===
        dice_names = ['dice1', 'dice2']
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        floor_name = 'floor'
        drawer_area_name = 'drawer-area'
        ready_pose_name = 'ready-pose'

        # === Exploration Phase: Identify Missing Predicate (drawer-closed) ===
        # The feedback suggests that the initial state may be missing the (drawer-closed drawer1) predicate.
        # We perform an exploration step to check the state of the drawer.
        # Since we cannot directly check predicates, we attempt to use available skills to infer the state.

        # Step 1: Move robot to the drawer area if not already there
        try:
            robot_pos = positions.get('robot', None)
            drawer_area_pos = positions.get(drawer_area_name, None)
            if drawer_area_pos is not None and robot_pos != drawer_area_pos:
                # Use execute_go to move robot
                obs, reward, done = execute_go(env, task, from_location=ready_pose_name, to_location=drawer_area_name)
                print(f"[Exploration] Robot moved to {drawer_area_name}")
        except Exception as e:
            print(f"[Exploration] Error moving robot to drawer area: {e}")

        # Step 2: Try to open the drawer (if not already open)
        # If the drawer is already open, execute_pull will fail or be unnecessary.
        # We attempt to pull the handle to check if the drawer is closed or open.
        try:
            # First, pick the handle if hand is empty
            obs, reward, done = execute_pick(env, task, obj=handle_name, p=drawer_area_name)
            print(f"[Exploration] Picked handle {handle_name}")
        except Exception as e:
            print(f"[Exploration] Could not pick handle {handle_name}: {e}")

        try:
            # Try to pull the handle to open the drawer
            obs, reward, done = execute_pull(env, task, d=drawer_name, h=handle_name, p=drawer_area_name)
            print(f"[Exploration] Pulled handle {handle_name} to open {drawer_name}")
        except Exception as e:
            print(f"[Exploration] Could not pull handle {handle_name}: {e}")
            print("[Exploration] Drawer may already be open or another predicate is missing.")

        # === Main Task Plan: Store Two Dice in the Open Drawer ===

        # For each dice: pick from floor, place in drawer
        for dice in dice_names:
            try:
                # Move to floor if not already there
                dice_pos = positions.get(dice, None)
                if dice_pos is not None and dice_pos != drawer_area_name:
                    obs, reward, done = execute_go(env, task, from_location=drawer_area_name, to_location=floor_name)
                    print(f"[Task] Robot moved to {floor_name} to pick {dice}")
            except Exception as e:
                print(f"[Task] Error moving to {floor_name} for {dice}: {e}")

            try:
                # Pick the dice from the floor
                obs, reward, done = execute_pick(env, task, obj=dice, p=floor_name)
                print(f"[Task] Picked {dice} from {floor_name}")
            except Exception as e:
                print(f"[Task] Could not pick {dice}: {e}")

            try:
                # Move to drawer area if not already there
                obs, reward, done = execute_go(env, task, from_location=floor_name, to_location=drawer_area_name)
                print(f"[Task] Robot moved to {drawer_area_name} to place {dice}")
            except Exception as e:
                print(f"[Task] Error moving to {drawer_area_name} for placing {dice}: {e}")

            try:
                # Place the dice in the open drawer
                obs, reward, done = execute_place(env, task, o=dice, d=drawer_name, p=drawer_area_name)
                print(f"[Task] Placed {dice} in {drawer_name}")
            except Exception as e:
                print(f"[Task] Could not place {dice} in {drawer_name}: {e}")

        # Optionally, close the drawer if required by the task
        try:
            obs, reward, done = execute_push(env, task, d=drawer_name, p=drawer_area_name)
            print(f"[Task] Closed {drawer_name}")
        except Exception as e:
            print(f"[Task] Could not close {drawer_name}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
