# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Retrieve object list from observation/init ===
        # Try to extract the object list from the observation or task description
        # Fallback: use keys from positions if not available
        object_list = []
        try:
            # Try to get from descriptions or obs if available
            if 'object-list' in descriptions:
                object_list = descriptions['object-list']
            elif 'object-list' in obs:
                object_list = obs['object-list']
            else:
                # Fallback: use all keys from positions that look like objects
                object_list = [k for k in positions.keys() if 'dice' in k or 'handle' in k or 'drawer' in k]
        except Exception:
            object_list = [k for k in positions.keys() if 'dice' in k or 'handle' in k or 'drawer' in k]

        # Helper function: check if object is in object_list
        def object_in_list(obj_name):
            return obj_name in object_list

        # === Exploration Phase: Identify missing predicates ===
        # Based on feedback, before attempting to manipulate objects, check if they exist in the object list.
        # If not, skip or report error.

        # Example objects and locations (adapt as needed)
        dice_names = ['dice1', 'dice2']
        drawer_names = ['drawer1', 'drawer2', 'drawer3']
        handle_names = ['handle1', 'handle2', 'handle3']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # For demonstration, try to put dice1 and dice2 into drawer1 (if possible)
        # 1. Check if dice1 and dice2 are in the object list
        for dice in dice_names:
            if not object_in_list(dice):
                print(f"[Exploration] {dice} not found in object list. Skipping.")
                continue

            # 2. Check if drawer1 is in the object list
            target_drawer = 'drawer1'
            if not object_in_list(target_drawer):
                print(f"[Exploration] {target_drawer} not found in object list. Skipping.")
                continue

            # 3. Check if handle1 is in the object list (for opening drawer1)
            target_handle = 'handle1'
            if not object_in_list(target_handle):
                print(f"[Exploration] {target_handle} not found in object list. Skipping.")
                continue

            # 4. Get positions for dice, drawer, handle, and locations
            try:
                dice_pos = positions[dice]
                drawer_pos = positions[target_drawer]
                handle_pos = positions[target_handle]
                # Use 'drawer-area' and 'ready-pose' as locations if available
                drawer_area_pos = positions.get('drawer-area', None)
                ready_pose_pos = positions.get('ready-pose', None)
                floor_pos = positions.get('floor', None)
            except Exception as e:
                print(f"[Error] Could not retrieve positions for {dice}, {target_drawer}, or {target_handle}: {e}")
                continue

            # 5. Move to dice location (assume on floor)
            try:
                print(f"[Task] Moving to {dice} at {dice_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='ready-pose',
                    to_location='floor',
                    max_steps=100
                )
            except Exception as e:
                print(f"[Error] execute_go to {dice} failed: {e}")
                continue

            # 6. Pick up the dice
            try:
                print(f"[Task] Picking up {dice}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=dice,
                    location='floor',
                    max_steps=100
                )
            except Exception as e:
                print(f"[Error] execute_pick for {dice} failed: {e}")
                continue

            # 7. Move to drawer area
            try:
                print(f"[Task] Moving to drawer area")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='floor',
                    to_location='drawer-area',
                    max_steps=100
                )
            except Exception as e:
                print(f"[Error] execute_go to drawer-area failed: {e}")
                continue

            # 8. Open the drawer if not already open
            # (Assume we need to pull the handle to open)
            try:
                print(f"[Task] Checking if {target_drawer} is open")
                # If drawer is not open, open it
                # (In a real implementation, check the state; here, always try)
                print(f"[Task] Picking up handle {target_handle}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=target_handle,
                    location='drawer-area',
                    max_steps=100
                )
                print(f"[Task] Pulling handle {target_handle} to open {target_drawer}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=target_drawer,
                    handle_name=target_handle,
                    location='drawer-area',
                    max_steps=100
                )
            except Exception as e:
                print(f"[Error] execute_pull for {target_drawer} failed: {e}")
                continue

            # 9. Place the dice in the drawer
            try:
                print(f"[Task] Placing {dice} in {target_drawer}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=dice,
                    drawer_name=target_drawer,
                    location='drawer-area',
                    max_steps=100
                )
            except Exception as e:
                print(f"[Error] execute_place for {dice} failed: {e}")
                continue

            # 10. Optionally, close the drawer after placing all dice
            # (Do this after the last dice)
            if dice == dice_names[-1]:
                try:
                    print(f"[Task] Pushing {target_drawer} to close it")
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name=target_drawer,
                        location='drawer-area',
                        max_steps=100
                    )
                except Exception as e:
                    print(f"[Error] execute_push for {target_drawer} failed: {e}")

        # === Exploration: Identify missing predicates ===
        # If any action fails due to a missing predicate (e.g., drawer not open, hand not empty, etc.),
        # print a message indicating which predicate might be missing.
        # This helps in debugging and improving the skill code.

        print("[Exploration] If you encountered errors above, check for missing predicates such as:")
        print(" - Object not in object list")
        print(" - Drawer not open before place")
        print(" - Hand not empty before pick")
        print(" - Drawer not unlocked before pull")
        print(" - Drawer not closed before pull")
        print(" - Drawer not full before place")
        print(" - Robot not at correct location before action")
        print(" - Handle not associated with drawer")
        print(" - etc.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
