# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # For this phase, we attempt to use the available skills to interact with all objects and drawers,
        # and observe if any action fails or if any state is not as expected, indicating a missing predicate.

        # Get all objects, drawers, and locations from the environment
        # (Assume positions dict contains keys for 'objects', 'drawers', 'handles', 'locations')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        handles = positions.get('handles', [])
        locations = positions.get('locations', [])

        # If not provided, try to infer from positions keys
        if not objects:
            objects = [k for k in positions.keys() if k.startswith('object')]
        if not drawers:
            drawers = [k for k in positions.keys() if k.startswith('drawer')]
        if not handles:
            handles = [k for k in positions.keys() if k.startswith('handle')]
        if not locations:
            locations = [k for k in positions.keys() if k.startswith('location') or k.startswith('room') or k.startswith('pos')]

        # Track which predicates we have explored
        explored_predicates = set()

        # 1. Try to pick up each object from the floor at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    explored_predicates.add('on-floor')
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}")
                        # Place it back if possible
                        for drawer in drawers:
                            try:
                                print(f"[Exploration] Trying to place {obj} in {drawer} at {loc}")
                                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                                explored_predicates.add('in-drawer')
                                if done:
                                    print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}")
                                    break
                            except Exception as e:
                                print(f"[Exploration] Place failed: {e}")
                        break
                except Exception as e:
                    print(f"[Exploration] Pick failed: {e}")

        # 2. Try to open and close each drawer using its handle
        for drawer in drawers:
            for handle in handles:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying to pick handle {handle} at {loc}")
                        obs, reward, done = execute_pick(env, task, handle, loc)
                        if done:
                            print(f"[Exploration] Picked handle {handle} at {loc}")
                            # Try to pull (open) the drawer
                            try:
                                print(f"[Exploration] Trying to pull (open) drawer {drawer} with handle {handle} at {loc}")
                                obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                                explored_predicates.add('drawer-open')
                                if done:
                                    print(f"[Exploration] Successfully opened drawer {drawer} at {loc}")
                                    # Try to push (close) the drawer
                                    try:
                                        print(f"[Exploration] Trying to push (close) drawer {drawer} at {loc}")
                                        obs, reward, done = execute_push(env, task, drawer, loc)
                                        explored_predicates.add('drawer-closed')
                                        if done:
                                            print(f"[Exploration] Successfully closed drawer {drawer} at {loc}")
                                    except Exception as e:
                                        print(f"[Exploration] Push failed: {e}")
                            except Exception as e:
                                print(f"[Exploration] Pull failed: {e}")
                    except Exception as e:
                        print(f"[Exploration] Pick handle failed: {e}")

        # 3. Try to move (go) between all pairs of locations
        for from_loc in locations:
            for to_loc in locations:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying to move from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        explored_predicates.add('robot-at')
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                    except Exception as e:
                        print(f"[Exploration] Move failed: {e}")

        # 4. Try to sweep each object at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    explored_predicates.add('swept')
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}")
                except Exception as e:
                    print(f"[Exploration] Sweep failed: {e}")

        # 5. Try to use gripper (if available)
        try:
            print(f"[Exploration] Trying to use gripper")
            obs, reward, done = execute_gripper(env, task)
            explored_predicates.add('robot-free')
            if done:
                print(f"[Exploration] Successfully used gripper")
        except Exception as e:
            print(f"[Exploration] Gripper failed: {e}")

        # 6. Report explored predicates
        print(f"[Exploration] Explored predicates: {explored_predicates}")

        # 7. If any action fails due to a missing predicate, log the error for further analysis

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()