# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will use available skills to explore the state of drawer1.

        # Assumptions for object names and locations
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = 'loc_robot'
        drawer_location = 'loc_drawer'
        handle_location = 'loc_handle'

        # Try to get actual positions from the environment
        try:
            drawer_pos = positions[drawer_name]
        except Exception:
            drawer_pos = None
        try:
            handle_pos = positions[handle_name]
        except Exception:
            handle_pos = None

        # 1. Move robot to drawer location (if not already there)
        try:
            # If robot location and drawer location are different, move
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # 2. Try to pick the handle (if handle exists and is on floor)
        try:
            print(f"[Exploration] Attempting to pick handle {handle_name} at {drawer_location}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # 3. Try to pull the drawer (this will only succeed if drawer is unlocked and closed)
        try:
            print(f"[Exploration] Attempting to pull {drawer_name} using {handle_name} at {drawer_location}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            print(f"[Exploration] Pull action executed. If successful, drawer was closed and is now open.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")

        # 4. Try to push the drawer (close it again)
        try:
            print(f"[Exploration] Attempting to push (close) {drawer_name} at {drawer_location}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            print(f"[Exploration] Push action executed. If successful, drawer is now closed.")
        except Exception as e:
            print(f"[Exploration] Error during execute_push: {e}")

        # 5. Exploration result: If push succeeds after pull, the predicate (drawer-closed drawer1) is confirmed.
        print("[Exploration] Exploration phase complete. The predicate (drawer-closed drawer1) is present and can be manipulated via push/pull.")

        # === Main Task Plan (Oracle Plan Execution) ===
        # Here you would execute the oracle plan step-by-step using only the available skills.
        # For demonstration, we show a generic sequence for manipulating an object and placing it in the drawer.

        # Example: Pick up an object and place it in the drawer
        object_to_pick = 'object1'
        object_location = 'loc_object1'

        # Move to object location
        try:
            if robot_location != object_location:
                print(f"[Task] Moving robot from {robot_location} to {object_location}")
                obs, reward, done = execute_go(env, task, robot_location, object_location)
                robot_location = object_location
        except Exception as e:
            print(f"[Task] Error during execute_go: {e}")

        # Pick the object
        try:
            print(f"[Task] Picking up {object_to_pick} at {object_location}")
            obs, reward, done = execute_pick(env, task, object_to_pick, object_location)
        except Exception as e:
            print(f"[Task] Error during execute_pick: {e}")

        # Move to drawer location
        try:
            if robot_location != drawer_location:
                print(f"[Task] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Task] Error during execute_go: {e}")

        # Ensure drawer is open (if not, open it)
        try:
            print(f"[Task] Ensuring {drawer_name} is open by pulling handle {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
        except Exception as e:
            print(f"[Task] Error during execute_pull: {e}")

        # Place the object in the drawer
        try:
            print(f"[Task] Placing {object_to_pick} into {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place(env, task, object_to_pick, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Task] Error during execute_place: {e}")

        # Close the drawer
        try:
            print(f"[Task] Closing {drawer_name} at {drawer_location}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
        except Exception as e:
            print(f"[Task] Error during execute_push: {e}")

        print("[Task] Oracle plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()