# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The missing predicate is likely related to the robot's location, specifically (robot-at drawer-area).
        # We will use the available skills to explore the environment and check if the predicate is recognized.

        # Step 1: Get all known locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'area' in name or 'location' in name or 'drawer' in name:
                location_names.append(name)
        # Fallback: If no area found, just use all keys
        if not location_names:
            location_names = list(positions.keys())

        # Step 2: Try to move the robot to each location using execute_go
        # and check if the environment recognizes the robot's presence at that location.
        # This will help us determine if (robot-at <location>) is a valid predicate.

        # For demonstration, we assume the robot starts at the first location.
        if len(location_names) < 2:
            print("[Exploration] Not enough locations to perform exploration.")
        else:
            from_location = location_names[0]
            to_location = location_names[1]
            print(f"[Exploration] Attempting to move robot from {from_location} to {to_location} using execute_go.")

            try:
                # Use the predefined skill execute_go
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_location,
                    to_location=to_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {to_location}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")

            # After moving, check if the predicate (robot-at to_location) is now true.
            # This can be done by querying the environment state or observation.
            # For demonstration, we print the observation.
            try:
                current_obs = task.get_observation()
                print(f"[Exploration] Current observation after move: {current_obs}")
            except Exception as e:
                print(f"[Exploration] Error retrieving observation: {e}")

        # Step 3: If the predicate is not recognized, try other available skills for exploration.
        # For example, try execute_sweep or execute_gripper to see if any new predicates are revealed.
        # This is a generic exploration step.
        for skill_name in ['execute_sweep', 'execute_gripper']:
            skill_fn = globals().get(skill_name, None)
            if skill_fn is not None:
                try:
                    print(f"[Exploration] Trying skill: {skill_name}")
                    obs, reward, done = skill_fn(
                        env,
                        task,
                        max_steps=50,
                        threshold=0.01,
                        timeout=5.0
                    )
                    print(f"[Exploration] Skill {skill_name} executed.")
                except Exception as e:
                    print(f"[Exploration] Error during {skill_name}: {e}")

        # Step 4: Print final observation to help identify missing predicates
        try:
            final_obs = task.get_observation()
            print(f"[Exploration] Final observation: {final_obs}")
        except Exception as e:
            print(f"[Exploration] Error retrieving final observation: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate information.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()