# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore all objects and drawers in the environment
        # For demonstration, we will iterate over all objects and drawers and attempt to use the skills
        # to see if any action fails due to a missing predicate (e.g., lock-known, identified, etc.)

        # Get all objects and drawers from positions (assuming keys are object/drawer names)
        object_names = [name for name in positions.keys() if 'object' in name or 'obj' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name or 'draw' in name]
        handle_names = [name for name in positions.keys() if 'handle' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'loc' in name or 'room' in name]

        # For robustness, collect all unique locations
        all_locations = set()
        for pos in positions.values():
            if isinstance(pos, dict) and 'location' in pos:
                all_locations.add(pos['location'])
            elif isinstance(pos, (tuple, list)) and len(pos) == 3:
                # Assume it's a 3D position, not a name
                continue
            elif isinstance(pos, str):
                all_locations.add(pos)
        all_locations = list(all_locations)
        if not all_locations:
            # Fallback: use location_names
            all_locations = location_names

        # Assume robot starts at the first location
        if all_locations:
            robot_location = all_locations[0]
        else:
            robot_location = None

        # --- Exploration: Try to pick and pull on all objects and drawers ---
        print("[Exploration] Starting exploration to identify missing predicates...")
        for obj in object_names:
            try:
                # Try to pick the object at its location
                obj_pos = positions[obj]
                # If position is a dict with 'location', use that
                if isinstance(obj_pos, dict) and 'location' in obj_pos:
                    obj_loc = obj_pos['location']
                elif isinstance(obj_pos, str):
                    obj_loc = obj_pos
                else:
                    # Fallback: use robot_location
                    obj_loc = robot_location

                print(f"[Exploration] Attempting to pick {obj} at {obj_loc}...")
                obs, reward, done = execute_pick(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully picked {obj}.")
                else:
                    print(f"[Exploration] Could not pick {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during pick({obj}): {e}")

            # Try to pull the object (if possible)
            try:
                print(f"[Exploration] Attempting to pull {obj} at {obj_loc}...")
                obs, reward, done = execute_pull(env, task, obj, obj, obj_loc)
                if done:
                    print(f"[Exploration] Successfully pulled {obj}.")
                else:
                    print(f"[Exploration] Could not pull {obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during pull({obj}): {e}")

        # Try to open/close drawers using handles
        for drawer in drawer_names:
            # Find handle for this drawer
            handle = None
            for h in handle_names:
                if drawer in h:
                    handle = h
                    break
            if not handle and handle_names:
                handle = handle_names[0]  # fallback

            # Try to pull (open) the drawer
            try:
                drawer_pos = positions[drawer]
                if isinstance(drawer_pos, dict) and 'location' in drawer_pos:
                    drawer_loc = drawer_pos['location']
                elif isinstance(drawer_pos, str):
                    drawer_loc = drawer_pos
                else:
                    drawer_loc = robot_location

                print(f"[Exploration] Attempting to pull (open) {drawer} with handle {handle} at {drawer_loc}...")
                obs, reward, done = execute_pull(env, task, drawer, handle, drawer_loc)
                if done:
                    print(f"[Exploration] Successfully pulled (opened) {drawer}.")
                else:
                    print(f"[Exploration] Could not pull (open) {drawer}.")
            except Exception as e:
                print(f"[Exploration] Exception during pull({drawer}): {e}")

            # Try to push (close) the drawer
            try:
                print(f"[Exploration] Attempting to push (close) {drawer} at {drawer_loc}...")
                obs, reward, done = execute_push(env, task, drawer, drawer_loc)
                if done:
                    print(f"[Exploration] Successfully pushed (closed) {drawer}.")
                else:
                    print(f"[Exploration] Could not push (close) {drawer}.")
            except Exception as e:
                print(f"[Exploration] Exception during push({drawer}): {e}")

        # Try to move (go) between locations
        if len(all_locations) >= 2:
            from_loc = all_locations[0]
            to_loc = all_locations[1]
            try:
                print(f"[Exploration] Attempting to go from {from_loc} to {to_loc}...")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
                else:
                    print(f"[Exploration] Could not move from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] Exception during go({from_loc}, {to_loc}): {e}")

        # Try to use other available skills (sweep, rotate, gripper)
        for skill in ['execute_sweep', 'execute_rotate', 'execute_gripper']:
            try:
                if skill == 'execute_sweep':
                    # Try to sweep each object at its location
                    for obj in object_names:
                        obj_pos = positions[obj]
                        if isinstance(obj_pos, dict) and 'location' in obj_pos:
                            obj_loc = obj_pos['location']
                        elif isinstance(obj_pos, str):
                            obj_loc = obj_pos
                        else:
                            obj_loc = robot_location
                        print(f"[Exploration] Attempting to sweep {obj} at {obj_loc}...")
                        obs, reward, done = execute_sweep(env, task, obj, obj_loc)
                elif skill == 'execute_rotate':
                    # If rotate takes no arguments
                    print(f"[Exploration] Attempting to rotate...")
                    obs, reward, done = execute_rotate(env, task)
                elif skill == 'execute_gripper':
                    print(f"[Exploration] Attempting to use gripper...")
                    obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print(f"[Exploration] Exception during {skill}: {e}")

        print("[Exploration] Exploration phase complete. Check above for any errors indicating missing predicates.")

        # === End of Exploration ===

        # TODO: After exploration, analyze the output/logs to determine which predicate is missing.
        # This can be done by reviewing the printed exceptions or failed actions above.

        # === (Optional) Main Task Plan Execution ===
        # After identifying the missing predicate, you would proceed to execute the oracle plan.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
