# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # Get all objects, drawers, and locations from positions (assuming keys are descriptive)
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # For robust exploration, try to use each skill on each object/drawer/location as appropriate
        # and catch exceptions if preconditions are not met.

        # 1. Try to pick up each object from the floor (execute_pick)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj,
                        loc
                    )
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # 2. Try to open each drawer (execute_pull) using each object as handle
        for drawer in drawer_names:
            for handle in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_pull on {drawer} with handle {handle} at {loc}")
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            drawer,
                            handle,
                            loc
                        )
                        if done:
                            print(f"[Exploration] Successfully pulled {drawer} with {handle} at {loc}")
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {loc}: {e}")

        # 3. Try to place each object into each drawer at each location (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_place for {obj} into {drawer} at {loc}")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj,
                            drawer,
                            loc
                        )
                        if done:
                            print(f"[Exploration] Successfully placed {obj} into {drawer} at {loc}")
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj} into {drawer} at {loc}: {e}")

        # 4. Try to push each drawer closed at each location (execute_push)
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer,
                        loc
                    )
                    if done:
                        print(f"[Exploration] Successfully pushed {drawer} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")

        # 5. Try to move between all pairs of locations (execute_go)
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_loc,
                            to_loc
                        )
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # 6. Try to sweep each object at each location (execute_sweep)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        obj,
                        loc
                    )
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully executed gripper")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()