# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will attempt to use available skills to explore the state of drawer1.

        # Assumptions for object names and locations
        # These should be replaced with actual names from your environment if different
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = None
        drawer_location = None
        handle_location = None

        # Try to find locations from positions dictionary
        try:
            if 'robot' in positions:
                robot_location = positions['robot']
            if drawer_name in positions:
                drawer_location = positions[drawer_name]
            if handle_name in positions:
                handle_location = positions[handle_name]
        except Exception as e:
            print("[Exploration] Error retrieving positions:", e)

        # If locations are not found, set to default or skip
        if robot_location is None:
            robot_location = [0, 0, 0]
        if drawer_location is None:
            drawer_location = [0.5, 0, 0]
        if handle_location is None:
            handle_location = [0.5, 0.1, 0]

        # Step 1: Move robot to drawer location if not already there
        try:
            # If robot is not at drawer location, move
            # We assume execute_go(from, to) is the correct skill
            # If the robot's current location is not known, we skip this step
            if 'execute_go' in globals():
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=drawer_location
                )
                print("[Exploration] Robot moved to drawer location.")
        except Exception as e:
            print("[Exploration] Error during execute_go:", e)

        # Step 2: Try to pick the handle (if available)
        try:
            if 'execute_pick' in globals():
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=handle_name,
                    location=drawer_location
                )
                print("[Exploration] Picked handle:", handle_name)
        except Exception as e:
            print("[Exploration] Error during execute_pick:", e)

        # Step 3: Try to pull the drawer (explore lock/closed state)
        try:
            if 'execute_pull' in globals():
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer_name,
                    handle=handle_name,
                    location=drawer_location
                )
                print("[Exploration] Pulled drawer to check state.")
        except Exception as e:
            print("[Exploration] Error during execute_pull:", e)

        # Step 4: If drawer is open, try to push it closed (to ensure (drawer-closed drawer1))
        try:
            if 'execute_push' in globals():
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=drawer_name,
                    location=drawer_location
                )
                print("[Exploration] Pushed drawer closed.")
        except Exception as e:
            print("[Exploration] Error during execute_push:", e)

        # Step 5: Exploration complete, print result
        print("[Exploration] Exploration phase complete. Checked for missing predicate (drawer-closed drawer1).")

        # === Main Task Plan (if any) ===
        # At this point, you can proceed with the main oracle plan if needed.
        # For this task, the focus is on exploration and predicate checking.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()