# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and drawers, and observe the effects.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to get all objects, drawers, locations from positions
        # This assumes positions is a dict like {'object_1': (x,y,z), ...}
        object_names = []
        drawer_names = []
        location_names = set()
        handle_names = []
        for name in positions:
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            else:
                object_names.append(name)
            # Try to extract location info if available
            pos = positions[name]
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
        # Fallback: if no explicit locations, use some default
        if not location_names:
            location_names = {'location_a', 'location_b', 'location_c'}

        # For demonstration, pick the first object, drawer, handle, and locations
        obj = object_names[0] if object_names else None
        drawer = drawer_names[0] if drawer_names else None
        handle = handle_names[0] if handle_names else None
        loc_list = list(location_names)
        loc = loc_list[0] if loc_list else 'location_a'
        loc2 = loc_list[1] if len(loc_list) > 1 else 'location_b'

        # === Exploration: Try each skill and observe for missing predicate ===
        # The goal is to see if any action fails due to a missing predicate, and log the error.

        print("[Exploration] Starting exploration to find missing predicate...")

        # Try execute_pick
        try:
            print(f"[Exploration] Trying execute_pick on object {obj} at {loc}")
            obs, reward, done = execute_pick(env, task, obj, loc)
            print("[Exploration] execute_pick succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        # Try execute_place
        try:
            print(f"[Exploration] Trying execute_place for object {obj} into drawer {drawer} at {loc}")
            obs, reward, done = execute_place(env, task, obj, drawer, loc)
            print("[Exploration] execute_place succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")

        # Try execute_pull (drawer open)
        try:
            print(f"[Exploration] Trying execute_pull on drawer {drawer} with handle {handle} at {loc}")
            obs, reward, done = execute_pull(env, task, drawer, handle, loc)
            print("[Exploration] execute_pull succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # Try execute_push (drawer close)
        try:
            print(f"[Exploration] Trying execute_push on drawer {drawer} at {loc}")
            obs, reward, done = execute_push(env, task, drawer, loc)
            print("[Exploration] execute_push succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        # Try execute_go (move robot)
        try:
            print(f"[Exploration] Trying execute_go from {loc} to {loc2}")
            obs, reward, done = execute_go(env, task, loc, loc2)
            print("[Exploration] execute_go succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # Try execute_sweep
        try:
            print(f"[Exploration] Trying execute_sweep on object {obj} at {loc}")
            obs, reward, done = execute_sweep(env, task, obj, loc)
            print("[Exploration] execute_sweep succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # Try execute_gripper
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # Try execute_rotate if available
        if 'execute_rotate' in available_skills:
            try:
                print(f"[Exploration] Trying execute_rotate on object {obj} at {loc}")
                obs, reward, done = execute_rotate(env, task, obj, loc)
                print("[Exploration] execute_rotate succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_rotate failed: {e}")

        print("[Exploration] Exploration phase complete. Check above for any failed actions and missing predicates.")

        # === End of Exploration ===

        # === (Optional) Main Task Plan Execution ===
        # If you have an oracle plan, you would execute it here step by step using the available skills.
        # For this task, the focus is on the exploration phase to identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()