# run_skeleton_task.py (Completed executable based on instructions, for predicate exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills directly, do not redefine

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        positions = get_object_positions()

        # ---- Exploration Phase: Try to trigger each predicate-exploring skill ----

        # For this exploration, we will attempt to:
        # - Move (execute_go) between every pair of known locations
        # - Pick each object at each location (execute_pick)
        # - Attempt to place, push, pull, sweep, etc.
        # The aim is to see which predicates are missing/required in the preconditions or effects.

        # Collect objects and locations from the positions dict (assumes structure {obj_name: (x, y, z, type)})
        all_objects = []
        all_locations = []
        object_to_type = {}

        for name, value in positions.items():
            # Assume value is tuple: (x, y, z, "type") or (x, y, z)
            if isinstance(value, tuple) and len(value) == 4:
                _, _, _, typ = value
                if typ == "location":
                    all_locations.append(name)
                else:
                    all_objects.append(name)
                    object_to_type[name] = typ
            elif isinstance(value, tuple) and len(value) == 3:
                # If type is missing, guess by name
                if name.startswith("loc") or name.startswith("room") or name.startswith("drawer"):
                    all_locations.append(name)
                else:
                    all_objects.append(name)
            else:
                continue  # skip unexpected

        # A robot can only be at one location -- let's try to enumerate pairs of locations
        tried_pairs = set()
        for from_loc in all_locations:
            for to_loc in all_locations:
                if from_loc != to_loc and (from_loc, to_loc) not in tried_pairs:
                    tried_pairs.add((from_loc, to_loc))
                    # Try moving
                    try:
                        print(f"[Explore] Trying move from {from_loc} to {to_loc}...")
                        obs, reward, done, info = execute_go(
                            env,
                            task,
                            start_location=from_loc,
                            end_location=to_loc
                        )
                        if done:
                            print(f"[Explore] Task ended after moving: {from_loc}->{to_loc}")
                            return
                    except Exception as e:
                        print(f"[Explore] Movement {from_loc} -> {to_loc} failed: {e}")
        
        # For each object at each location – try execute_pick (if hand is empty, robot is free, object is on floor)
        for obj in all_objects:
            for loc in all_locations:
                try:
                    print(f"[Explore] Trying pick up {obj} at {loc} ...")
                    obs, reward, done, info = execute_pick(
                        env,
                        task,
                        target_object=obj,
                        location=loc
                    )
                    if done:
                        print(f"[Explore] Task ended after picking: {obj} at {loc}")
                        return
                except Exception as e:
                    print(f"[Explore] Pick {obj} at {loc} failed: {e}")
        
        # For each drawer, attempt to pull (open) and push (close)
        # Try to identify drawers/handles heuristically
        for drawer in all_locations:
            if "drawer" in drawer:
                handles = [obj for obj in all_objects if "handle" in obj and drawer in obj]
                handle = handles[0] if handles else None
                # Try execute_pull (requires a handle object), robot holding handle, drawer unlocked and closed, etc.
                if handle:
                    try:
                        print(f"[Explore] Trying pick handle {handle} at {drawer} ...")
                        obs, reward, done, info = execute_pick(
                            env,
                            task,
                            target_object=handle,
                            location=drawer
                        )
                        if done:
                            print(f"[Explore] Task ended when picking handle: {handle} at {drawer}")
                            return
                        print(f"[Explore] Trying execute_pull on {drawer} with {handle} ...")
                        obs, reward, done, info = execute_pull(
                            env,
                            task,
                            drawer=drawer,
                            handle=handle,
                            location=drawer
                        )
                        if done:
                            print(f"[Explore] Task ended after pulling: {drawer} with {handle}")
                            return
                    except Exception as e:
                        print(f"[Explore] Pull {drawer} with {handle} failed: {e}")
                # try to push closed drawer (even if not open, to see if error is predicate-based)
                try:
                    print(f"[Explore] Trying to push drawer {drawer} ...")
                    obs, reward, done, info = execute_push(
                        env,
                        task,
                        drawer=drawer,
                        location=drawer
                    )
                    if done:
                        print(f"[Explore] Task ended after pushing: {drawer}")
                        return
                except Exception as e:
                    print(f"[Explore] Push {drawer} failed: {e}")

        # Try sweeping objects at their locations
        for obj in all_objects:
            for loc in all_locations:
                try:
                    print(f"[Explore] Trying to sweep {obj} at {loc} ...")
                    obs, reward, done, info = execute_sweep(
                        env,
                        task,
                        target_object=obj,
                        location=loc
                    )
                    if done:
                        print(f"[Explore] Task ended after sweeping: {obj} at {loc}")
                        return
                except Exception as e:
                    print(f"[Explore] Sweep {obj} at {loc} failed: {e}")

        # Try rotating and gripper action for completeness
        try:
            print(f"[Explore] Trying to use gripper...")
            obs, reward, done, info = execute_gripper(env, task)
        except Exception as e:
            print(f"[Explore] Gripper action failed: {e}")
        try:
            print(f"[Explore] Trying to use rotate...")
            obs, reward, done, info = execute_rotate(env, task)
        except Exception as e:
            print(f"[Explore] Rotate action failed: {e}")

        print("[Explore] Exploration phase complete.")
        # If any of the above trigger missing predicates in errors or state, print finds here.

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
