# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined primitives as per instruction

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Run a generic task with predicate exploration for missing predicate diagnosis.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers
        init_video_writers(obs)

        # Wrap step function for video recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        print("[Exploration] Acquired object positions:", positions)
        
        # == Exploration Phase ==
        # The goal is to detect missing predicates by interacting with environment via predefined skills
        # and observing reactions (e.g., failing actions, unexpected state transitions).
        
        # Sample standardized names, will need mapping based on env's actual names!
        robot_location = None
        all_locations = []
        all_drawers = []
        all_objects = []
        all_handles = []

        # Try to parse positions dict to divide location/object/drawer types by PDDL types or known suffixes
        for name, pos in positions.items():
            lname = name.lower()
            if 'drawer' in lname:
                all_drawers.append(name)
            elif 'handle' in lname:
                all_handles.append(name)
            elif 'loc' in lname or 'room' in lname or 'table' in lname:
                all_locations.append(name)
            else:
                all_objects.append(name)
        # If not detected, fallback to all keys for objects
        if len(all_objects) == 0:
            all_objects = [name for name in positions.keys()]

        # Try to get robot initial location (pick first location as heuristic if not available)
        if hasattr(task, "get_robot_location"):
            robot_location = task.get_robot_location()
            if isinstance(robot_location, list):  # In case function returns [name, ...]
                robot_location = robot_location[0]
        elif len(all_locations) > 0:
            robot_location = all_locations[0]

        # If not found, try from observations (robustness fallback, e.g., RLBench's descriptions)
        if robot_location is None and isinstance(descriptions, dict):
            for k, v in descriptions.items():
                if 'robot-at' in k or 'robot_at' in k or 'robot_location' in k:
                    robot_location = v

        if not robot_location:
            print("[Exploration] WARNING: Robot starting location not found, using first available location if any.")
            robot_location = all_locations[0] if len(all_locations) > 0 else None

        print("[Exploration] Robot initial location:", robot_location)
        print("[Exploration] Detected locations:", all_locations)
        print("[Exploration] Detected drawers:", all_drawers)
        print("[Exploration] Detected objects:", all_objects)
        print("[Exploration] Detected handles:", all_handles)

        # === Predicate Discovery by Exploration ===

        # 1. Try to 'execute_pick' each object at its position (if 'on-floor' is assumed, as in PDDL)
        # Try to catch which precondition fails (e.g., missing 'on-floor' predicate, missing 'hand-empty', etc.)
        detected_failures = []
        for obj in all_objects:
            try:
                print(f"[Exploration] Attempting to pick object {obj} at {robot_location} ...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=obj,
                    p=robot_location
                )
                print(f"[Exploration] Pick succeeded for {obj}: reward={reward}, done={done}")
                # Check holding status, etc., if available
                break  # Stop after first success, since one probe is enough
            except Exception as e:
                print(f"[Exploration] Pick failed for {obj}: {str(e)}")
                detected_failures.append(("pick", obj, str(e)))
                continue

        # 2. Try to 'execute_go' to a different location (movements)
        for to_location in all_locations:
            if to_location == robot_location:
                continue
            try:
                print(f"[Exploration] Attempting to move robot from {robot_location} to {to_location} ...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_=robot_location,
                    to=to_location
                )
                print(f"[Exploration] Move succeeded: reward={reward}, done={done}")
                robot_location = to_location
                break
            except Exception as e:
                print(f"[Exploration] Move failed: {str(e)}")
                detected_failures.append(("go", (robot_location, to_location), str(e)))
                continue

        # 3. Try to 'execute_pull' on each drawer's handle if both handles and drawers are detected
        if all_handles and all_drawers:
            for d in all_drawers:
                for h in all_handles:
                    try:
                        print(f"[Exploration] Attempting to pull drawer {d} using handle {h} at {robot_location} ...")
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            d=d,
                            h=h,
                            p=robot_location
                        )
                        print(f"[Exploration] Pull succeeded for drawer {d}: reward={reward}, done={done}")
                        break
                    except Exception as e:
                        print(f"[Exploration] Pull failed for drawer {d} with handle {h}: {str(e)}")
                        detected_failures.append(("pull", (d, h), str(e)))
                        continue

        # 4. Try to 'execute_place' any held object into any open drawer at the current location
        # This also tests predicates for 'drawer-open', 'not drawer-full', etc.
        # Here, we assume only one object is held at a time
        for d in all_drawers:
            for obj in all_objects:
                try:
                    print(f"[Exploration] Attempting to place {obj} in drawer {d} at {robot_location} ...")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        o=obj,
                        d=d,
                        p=robot_location
                    )
                    print(f"[Exploration] Place succeeded for {obj} in {d}: reward={reward}, done={done}")
                    break
                except Exception as e:
                    print(f"[Exploration] Place failed for {obj} in {d}: {str(e)}")
                    detected_failures.append(("place", (obj, d), str(e)))
                    continue

        # 5. Log all failed attempts for predicate diagnosis
        if detected_failures:
            print("[Exploration] Failure summary (possible missing resources/predicates):")
            for action_name, params, error_msg in detected_failures:
                print(f"    Action {action_name} with {params} failed: {error_msg}")

        # You can now, from the action failures and their error messages,
        # infer which PDDL predicates may be missing, such as 'on-floor', 'drawer-open', etc.
        print("[Exploration] Exploration phase completed.")
        
        # You can optionally proceed to the true task plan with knowledge about predicate support.
        # (Insert oracle plan execution logic here.)

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
