# run_skeleton_task.py (Completed for Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine primitives; use as provided.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with an exploration phase to find missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Use get_object_positions() to get object info, e.g.:
        positions = get_object_positions()
        # Assume keys for common objects: 'drawer1', 'handle1', etc.
        # Update variable names if necessary based on your environment.

        # === Exploration Phase ===
        # Based on feedback: need to determine if (drawer-closed drawer1) is true/missing
        # We'll use available skills to systematically check/fill predicates.
        # Assumption: We want to see if drawer1 is closed (by direct or indirect action).
        # Use execute_pull/explore to check 'lock-known' and 'closed' status.

        print("[Exploration] Starting predicate exploration for drawer1.")

        # We'll try to use the available skills to interact with the drawer and its handle.
        # Typical flow (update names/keys as needed for your env/plan):
        try:
            # Check positions
            drawer_name = 'drawer1'
            handle_name = None
            for key in positions.keys():
                if key.startswith('handle'):
                    handle_name = key
                    break
            if handle_name is None:
                raise ValueError("[Exploration] Could not find handle object in positions.")

            drawer_pos = positions[drawer_name]
            handle_pos = positions[handle_name]

            # Step 1: Move robot to drawer position (assume robot location is part of positions)
            robot_current_loc = None
            for k in positions.keys():
                if k.startswith('robot'):
                    robot_current_loc = k
                    break
            if robot_current_loc is None:
                raise ValueError("[Exploration] Could not detect robot's current location.")

            robot_loc = positions[robot_current_loc]

            # Step 2: Use execute_go to move robot to handle location
            print(f"[Exploration] Moving robot from {robot_current_loc} to handle location for exploration.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_current_loc,
                to_location=handle_name
            )

            # Step 3: Try picking the handle
            print(f"[Exploration] Attempting to pick handle: {handle_name} at location.")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=handle_name,
                location=handle_name
            )

            # Step 4: Attempt to pull the drawer (if skill available)
            print(f"[Exploration] Trying to pull (open) the drawer {drawer_name}.")
            obs, reward, done = execute_pull(
                env,
                task,
                d=drawer_name,
                h=handle_name,
                p=handle_name
            )

            # If success, the drawer has been opened, which means it was initially closed.
            print(f"[Exploration] Drawer {drawer_name} successfully pulled. It was initially closed.")

            # Alternatively, error/exception would mean drawer wasn't closed, which is unexpected.

        except Exception as e:
            print(f"[Exploration] Exception during predicate exploration: {e}")

        # === Oracle Plan Execution Phase ===
        # Here you would process the oracle plan to bring the robot to the goal state.
        # For demonstration, we assume the goal involves picking and placing an object into the drawer.
        # (Replace with the actual oracle plan for your task.)

        print("[Task Phase] Executing oracle plan steps...")

        try:
            # Example plan:
            # 1. Move to location of object to pick (on floor).
            object_on_floor = None
            object_loc = None
            for key, pos in positions.items():
                if key.startswith('obj') or key.startswith('object'):
                    object_on_floor = key
                    object_loc = key
                    break
            if object_on_floor is not None:
                print(f"[Task] Moving to {object_loc} to pick up {object_on_floor}.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=handle_name,  # last known robot location
                    to_location=object_loc
                )

                # 2. Pick the object.
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=object_on_floor,
                    location=object_loc
                )
                print(f"[Task] Picked up {object_on_floor}.")

                # 3. Move to drawer to place object.
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=object_loc,
                    to_location=drawer_name
                )

                # 4. Place object into the now-open drawer.
                obs, reward, done = execute_place(
                    env,
                    task,
                    o=object_on_floor,
                    d=drawer_name,
                    p=drawer_name
                )
                print(f"[Task] Placed {object_on_floor} into {drawer_name}.")

            else:
                print("[Task] No object found on floor to pick.")

        except Exception as e:
            print(f"[Task Phase] Exception during oracle plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
