# run_skeleton_task.py (Executable Robot Control Plan for Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import * # Use provided primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Predicate Exploration Phase ===
        print("[Exploration] Starting predicate exploration to find missing predicates...")

        # For illustration, attempt all relevant actions to detect which predicate is missing
        # We try the primitives in various reasonable combinations to elicit missing-predicate errors

        # Attempt to pick up all objects on the floor (for 'on-floor' detection etc.)
        if 'execute_pick' in globals():
            try:
                for obj_name, obj_info in positions.items():
                    if obj_info.get('location_type') == 'floor':
                        print(f"[Exploration] Trying to pick object '{obj_name}' at {obj_info['location']} ...")
                        # Skill API: (env, task, obj_name, location)
                        obs, reward, done = execute_pick(
                            env, 
                            task, 
                            obj_name, 
                            obj_info['location']
                        )
                        if done:
                            print(f"[Exploration] Picked {obj_name} -- terminating predicate search early.")
                            break
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick of object '{obj_name}': {e}")

        # Attempt to open all drawers (for 'drawer-open', 'drawer-closed', 'drawer-unlocked' etc.)
        if 'execute_pull' in globals():
            try:
                for drawer, attrs in positions.items():
                    if attrs.get('type') == 'drawer' and attrs.get('handle'):
                        print(f"[Exploration] Trying to open drawer '{drawer}' with handle '{attrs['handle']}' ...")
                        # Try picking the handle first (most drawers require handle to be held)
                        try:
                            obs, reward, done = execute_pick(
                                env, 
                                task, 
                                attrs['handle'], 
                                attrs['location']
                            )
                        except Exception as e:
                            print(f"[Exploration] Cannot pick handle '{attrs['handle']}': {e}")

                        # Now attempt to pull
                        try:
                            obs, reward, done = execute_pull(
                                env,
                                task,
                                drawer,
                                attrs['handle'],
                                attrs['location']
                            )
                        except Exception as e:
                            print(f"[Exploration] Exception during execute_pull on drawer '{drawer}': {e}")
            except Exception as e:
                print(f"[Exploration] Exception in explore-pull phase: {e}")

        # Attempt to place objects in open drawers (for 'drawer-full', 'drawer-partially-filled')
        if 'execute_place' in globals():
            try:
                for drawer, d_attrs in positions.items():
                    if d_attrs.get('type') == 'drawer' and d_attrs.get('is_open', False):
                        # Find a held object to place, or pick one up
                        for obj_name, obj_info in positions.items():
                            if obj_info.get('location_type') == 'held' or obj_info.get('location_type') == 'floor':
                                try:
                                    # If not holding, pick first
                                    if obj_info.get('location_type') == 'floor':
                                        print(f"[Exploration] Picking up '{obj_name}' for placing.")
                                        obs, reward, done_pick = execute_pick(
                                            env, 
                                            task, 
                                            obj_name, 
                                            obj_info['location']
                                        )
                                    print(f"[Exploration] Trying to place '{obj_name}' in drawer '{drawer}'.")
                                    obs, reward, done = execute_place(
                                        env, 
                                        task, 
                                        obj_name, 
                                        drawer, 
                                        d_attrs['location']
                                    )
                                    if done:
                                        print(f"[Exploration] Placed {obj_name} in {drawer}.")
                                        break
                                except Exception as e:
                                    print(f"[Exploration] Exception during execute_place: {e}")
            except Exception as e:
                print(f"[Exploration] Exception in explore-place phase: {e}")

        # Attempt to push (close) open drawers
        if 'execute_push' in globals():
            try:
                for drawer, attrs in positions.items():
                    if attrs.get('type') == 'drawer' and attrs.get('is_open', False):
                        print(f"[Exploration] Trying to push (close) drawer '{drawer}'.")
                        try:
                            obs, reward, done = execute_push(
                                env, 
                                task, 
                                drawer, 
                                attrs['location']
                            )
                        except Exception as e:
                            print(f"[Exploration] Exception during execute_push: {e}")
            except Exception as e:
                print(f"[Exploration] Exception in explore-push phase: {e}")

        # Attempt simple movement between known locations (for predicates like 'robot-at')
        if 'execute_go' in globals():
            loc_names = [v['location'] for v in positions.values() if 'location' in v]
            unique_locs = list(set(loc_names))
            if len(unique_locs) > 1:
                try:
                    print(f"[Exploration] Trying to move from '{unique_locs[0]}' to '{unique_locs[1]}'.")
                    obs, reward, done = execute_go(
                        env, 
                        task, 
                        unique_locs[0], 
                        unique_locs[1]
                    )
                except Exception as e:
                    print(f"[Exploration] Exception during execute_go: {e}")

        # Extra: Try sweep/rotate/gripper skills if available for further coverage
        if 'execute_sweep' in globals():
            for obj_name, obj_info in positions.items():
                if obj_info.get('location_type') == 'floor':
                    try:
                        print(f"[Exploration] Trying to sweep object '{obj_name}' at {obj_info['location']}.")
                        obs, reward, done = execute_sweep(
                            env,
                            task,
                            obj_name,
                            obj_info['location']
                        )
                    except Exception as e:
                        print(f"[Exploration] Exception during execute_sweep: {e}")

        if 'execute_gripper' in globals():
            try:
                print("[Exploration] Trying gripper execute_gripper().")
                obs, reward, done = execute_gripper(env, task)
            except Exception as e:
                print(f"[Exploration] Exception during execute_gripper: {e}")

        # ======= End of Exploration Phase ======
        print("[Exploration] Exploration phase ended. Check logs above for clues regarding missing predicates.")
        print("If an action fails or gives an error mentioning a condition or predicate,")
        print("it is likely that the predicate mentioned is missing or not established in the initial state.")
        
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()